#define DBG  print*,__FILE__,__LINE__
#define OUTTXTPRF
!-------------------------------------------------------------------------------------------------
!BOP
! !MODULE: ocdprf_mod
! !DESCRIPTION: 
!  This module contains the routines for ocean data input and output. The observation
!  contains 3 kinds: 1) Profiles, like the temperature/salinity profiles from Argo, EN4
!  or GTSPP.
!\\\\
! !REVISION HISTORY:
!  2017/10/11, Xunqiang Yin, First Institute of Oceanography, SOA, China
! !REMARKS:
! 2017/10/11
! !INTERFACE: 
  module ocndata_mod
! !USES:
  use time_mod
  use netcdf_mod
!EOP
!-------------------------------------------------------------------------------------------------
  implicit none
!BOP
! !PUBLIC MEMBER FUNCTIONS:

  public :: input_profs
  public :: fixedprofs
  public :: get_prof_no
  public :: get_prof_data
  public :: set_prof_flnm
  public :: out_prof_data
  public :: out_oneprof_data
  public :: out_prof_data_asci
  public :: release_ocndat

  public :: release_fld2d
  public :: outfld2d_nc,inpfld2d_nc,inpfld2d_nc_var
  public :: outfld2d,inpfld2d

  public :: releasesct
  public :: outsct_nc
  public :: inputsct_nc
  public :: outsct_nc_append

  public :: output_sctdata
  public :: readin_sctobs
  public :: output_sctobs

  public :: releasesct3d
  public :: inputsct3d_nc
  public :: outsct3d_nc
  
  private

! !PUBLIC DATA MEMBERS:

  real(4),parameter,public :: miss_val=9.9692099683868690e+36
  integer,parameter,public :: ocnobsvid_temp=1
  integer,parameter,public :: ocnobsvid_salt=2
  integer,parameter,public :: ocnobsvid_ssh =3
  integer,parameter,public :: ocnobsvid_sst =4
  integer,parameter,public :: ocnobsvid_sla =5
  integer,parameter,public :: ocnobsvid_potm=6
  
! !PUBLIC TYPES:

  type,public :: ocndata_dat_type
    integer         :: nv=0           ! - Number of variables.
    integer         :: isize          ! - Size of data in X.
    integer         :: jsize          ! - Size of data in Y.
    integer         :: ksize          ! - Size of data in Z.
    real(4),pointer :: x(:)           ! - X location of data.
    real(4),pointer :: y(:)           ! - Y location of data.
    real(4),pointer :: z(:)           ! - Z location of data.
    real(4),pointer :: var(:,:,:,:)   ! - Variable values.(isize,jsize,ksize,nv)
    integer,pointer :: varid(:)       ! - Variable IDs.(nv)
    integer,pointer :: flag(:)        ! - Flags for data.(2)
    real(4)         :: missv=miss_val ! - Missing value of Data.
    real(8)         :: time           ! - Time of data.
  end type ocndata_dat_type

  type,public :: ocndata_profs_type
    integer :: n                                 ! Number of ocean observed data.
    type(ocndata_dat_type),pointer :: ocndat(:)  ! Ocean observed data sets (n)
  end type ocndata_profs_type

!-------------------------------------------------------------------------------
  type,public :: fld2d_var_type
    integer           :: flag    = 0
    integer           :: nodata  = 1
    integer           :: im      = 0
    integer           :: jm      = 0
    real(4)           :: misv    = -1.e10
    character(len=14) :: ctime   =''
    character(len=14) :: vname   =''
    character(len=14) :: xname   =''
    character(len=14) :: yname   =''
    character(len=80) :: cflag   =''
    real(4),pointer   :: lon(:)  =>null()
    real(4),pointer   :: lat(:)  =>null()
    real(4),pointer   :: var(:,:)=>null()
    real(4),pointer   :: err(:,:)=>null()
  end type fld2d_var_type
!-------------------------------------------------------------------------------
  type,public :: sct2d_type
    integer                   :: n        =0       ! number of records
    integer                   :: nv       =0       ! number of variables
    character(len=14)         :: cflag    =''      ! flag for this dataset.
    character(len=90),pointer :: lname(:) =>null() ! (nv), name of variables
    character(len=20),pointer :: vname(:) =>null() ! (nv), name of variables
    real(8)          ,pointer :: x(:)     =>null() ! (n), longitude, degree east, 0-360
    real(8)          ,pointer :: y(:)     =>null() ! (n), latitude, degree north, -90-90
    real(8)          ,pointer :: t(:)     =>null() ! (n), time in days since 1950-1-1
    real(8)          ,pointer :: v(:,:)   =>null() ! (n,nv), values of variables
  end type sct2d_type
!-------------------------------------------------------------------------------
  type,public :: sct3d_type
    integer                   :: n        =0       ! number of records
    integer                   :: nv       =0       ! number of variables
    character(len=14)         :: cflag    =''      ! flag for this dataset.
    character(len=99),pointer :: lname(:) =>null() ! (nv), name of variables
    character(len=20),pointer :: vname(:) =>null() ! (nv), name of variables
    real(8)          ,pointer :: x(:)     =>null() ! (n), longitude, degree east, 0-360
    real(8)          ,pointer :: y(:)     =>null() ! (n), latitude, degree north, -90-90
    real(8)          ,pointer :: z(:)     =>null() ! (n), latitude, degree north, -90-90
    real(8)          ,pointer :: t(:)     =>null() ! (n), time in days since 1950-1-1
    real(8)          ,pointer :: v(:,:)   =>null() ! (n,nv), values of variables
  end type sct3d_type
!-------------------------------------------------------------------------------
!EOP
  type(ocndata_profs_type) :: fixedprofs
!-------------------------------------------------------------------------------
  contains
!-------------------------------------------------------------------------------------------------

  subroutine outsct3d_nc(path,sct)
    character(len=*),intent(in) :: path
    type(sct3d_type),intent(in) :: sct
    integer :: ncid,iv,i,start_recd
    character(len=256) :: filename,patho
    character :: slash='/'
    character(len=14) :: ctime
    logical :: ext
    if(sct%n<1)return
    i=len_trim(path);if(path(i:i)==slash)slash=' '
    ctime=datestr(sct%t(1))
    patho=trim(path)//trim(slash)//ctime(1:4)//'/'
    call system('mkdir -p '//trim(patho))
    filename=trim(patho)//trim(sct%cflag)//'.sct3d.'//ctime(1:8)//'.nc'
    inquire(file=filename,exist=ext)
    if(ext)then
      call open_nc(ncid,filename,'w')
      start_recd=get_dimension_len(ncid,'recd')+1
    else
      call open_nc(ncid,filename,'c')
      call dimension_define(ncid,'recd',0)
      call variable_define(ncid,'time',nf_double,['recd'])
      call variable_define(ncid,'lon',nf_real,['recd'])
      call variable_define(ncid,'lat',nf_real,['recd'])
      call variable_define(ncid,'dep',nf_real,['recd'])
      do iv=1,sct%nv
        call variable_define(ncid,sct%vname(iv),nf_real,['recd'])
      enddo
      if(associated(sct%lname))then
        do iv=1,sct%nv
          call set_attribute(ncid,'longname',sct%lname(iv),sct%vname(iv))
        enddo
      endif
      call end_define(ncid)
      start_recd=1
    endif
    call writenc(ncid,'lon',sct%x,start_recd)
    call writenc(ncid,'lat',sct%y,start_recd)
    call writenc(ncid,'dep',sct%z,start_recd)
    call writenc(ncid,'time',sct%t,start_recd)
    do iv=1,sct%nv
      call writenc(ncid,sct%vname(iv),sct%v(:,iv),start_recd)
    enddo
    call close_nc(ncid)
  end subroutine outsct3d_nc

  subroutine inputsct3d_nc(path,ctime,limits,sct)
    character(len=*),intent(in) :: path,ctime
    real(4),intent(in) :: limits(4)
    type(sct3d_type),intent(inout) :: sct
    integer :: ncid,iv,i,rec
    character(len=256) :: filename,patho
    character :: slash='/'

    i=len_trim(path);if(path(i:i)==slash)slash=' '
    patho=trim(path)//trim(slash)//ctime(1:4)//'/'
    filename=trim(patho)//trim(sct%cflag)//'.sct3d.'//ctime(1:8)//'.nc'

    call open_nc(ncid,filename,'r')
    sct%n=get_dimension_len(ncid,'recd')
    allocate(sct%x(sct%n),sct%y(sct%n),sct%z(sct%n),sct%t(sct%n),sct%v(sct%n,sct%nv))
    call readnc(ncid,'lon',sct%x)
    call readnc(ncid,'lat',sct%y)
    call readnc(ncid,'dep',sct%z)
    call readnc(ncid,'time',sct%t)
    do iv=1,sct%nv
      call readnc(ncid,sct%vname(iv),sct%v(:,iv))
    enddo
    call close_nc(ncid)
    write(*,*)'minval(sct%x)',minval(sct%x),limits(1)
    write(*,*)'maxval(sct%x)',maxval(sct%x),limits(2)
    write(*,*)'minval(sct%y)',minval(sct%y),limits(3)
    write(*,*)'maxval(sct%y)',maxval(sct%y),limits(4)
    rec=0
    do i=1,sct%n
      if(sct%x(i)<limits(1))cycle
      if(sct%x(i)>limits(2))cycle
      if(sct%y(i)<limits(3))cycle
      if(sct%y(i)>limits(4))cycle
      rec=rec+1
      sct%x(rec)=sct%x(i)
      sct%y(rec)=sct%y(i)
      sct%z(rec)=sct%z(i)
      sct%t(rec)=sct%t(i)
      sct%v(rec,:)=sct%v(i,:)
    enddo
    sct%n=rec
  end subroutine inputsct3d_nc

  subroutine releasesct3d(sct)
    type(sct3d_type),intent(inout) :: sct
    if(associated(sct%vname))then;deallocate(sct%vname);nullify(sct%vname);endif
    if(associated(sct%v    ))then;deallocate(sct%v    );nullify(sct%v    );endif
    if(associated(sct%x    ))then;deallocate(sct%x    );nullify(sct%x    );endif
    if(associated(sct%y    ))then;deallocate(sct%y    );nullify(sct%y    );endif
    if(associated(sct%t    ))then;deallocate(sct%t    );nullify(sct%t    );endif
  end subroutine releasesct3d
  
  subroutine releasesct(sct)
    type(sct2d_type),intent(inout) :: sct
    if(associated(sct%vname))then;deallocate(sct%vname);nullify(sct%vname);endif
    if(associated(sct%v    ))then;deallocate(sct%v    );nullify(sct%v    );endif
    if(associated(sct%x    ))then;deallocate(sct%x    );nullify(sct%x    );endif
    if(associated(sct%y    ))then;deallocate(sct%y    );nullify(sct%y    );endif
    if(associated(sct%t    ))then;deallocate(sct%t    );nullify(sct%t    );endif
  end subroutine releasesct
      
!-------------------------------------------------------------------------------------------------

  subroutine inputsct_nc(path,ctime,limits,sct)
    character(len=*),intent(in) :: path,ctime
    real(4),intent(in) :: limits(4)
    type(sct2d_type),intent(inout) :: sct
    integer :: ncid,iv,i,rec
    character(len=256) :: filename,patho
    character :: slash='/'
    i=len_trim(path);if(path(i:i)==slash)slash=' '
    patho=trim(path)//trim(slash)//ctime(1:4)//'/'
    filename=trim(patho)//trim(sct%cflag)//'.sct.'//ctime(1:8)//'.nc'
    call open_nc(ncid,filename,'r')
    sct%n=get_dimension_len(ncid,'recd')
    allocate(sct%x(sct%n),sct%y(sct%n),sct%t(sct%n),sct%v(sct%n,sct%nv))
    call readnc(ncid,'lon',sct%x)
    call readnc(ncid,'lat',sct%y)
    call readnc(ncid,'time',sct%t)
    do iv=1,sct%nv
      call readnc(ncid,sct%vname(iv),sct%v(:,iv))
    enddo
    call close_nc(ncid)
    !write(*,*)'minval(sct%x)',minval(sct%x),limits(1)
    !write(*,*)'maxval(sct%x)',maxval(sct%x),limits(2)
    !write(*,*)'minval(sct%y)',minval(sct%y),limits(3)
    !write(*,*)'maxval(sct%y)',maxval(sct%y),limits(4)
    rec=0
    do i=1,sct%n
      if(sct%x(i)<limits(1))cycle
      if(sct%x(i)>limits(2))cycle
      if(sct%y(i)<limits(3))cycle
      if(sct%y(i)>limits(4))cycle
      rec=rec+1
      sct%x(rec)=sct%x(i)
      sct%y(rec)=sct%y(i)
      sct%t(rec)=sct%t(i)
      sct%v(rec,:)=sct%v(i,:)
    enddo
    sct%n=rec
  end subroutine inputsct_nc
  
  subroutine outsct_nc(path,sct)
    character(len=*),intent(in) :: path
    type(sct2d_type),intent(in) :: sct
    integer :: ncid,iv,i,start_recd
    character(len=256) :: filename,patho
    character :: slash='/'
    character(len=14) :: ctime
    logical :: ext
    if(sct%n<1)return
    i=len_trim(path);if(path(i:i)==slash)slash=' '

    ctime=datestr(sct%t(1))
    patho=trim(path)//trim(slash)//ctime(1:4)//'/'
    call system('mkdir -p '//trim(patho))
    filename=trim(patho)//trim(sct%cflag)//'.sct.'//ctime(1:8)//'.nc'
    inquire(file=filename,exist=ext)
    if(ext)then
      call open_nc(ncid,filename,'w')
      start_recd=get_dimension_len(ncid,'recd')+1
    else
      call open_nc(ncid,filename,'c')
      call dimension_define(ncid,'recd',0)
      call variable_define(ncid,'time',nf_double,['recd'])
      call variable_define(ncid,'lon',nf_real,['recd'])
      call variable_define(ncid,'lat',nf_real,['recd'])
      do iv=1,sct%nv
        call variable_define(ncid,sct%vname(iv),nf_real,['recd'])
      enddo
      if(associated(sct%lname))then
        do iv=1,sct%nv
          call set_attribute(ncid,'longname',sct%lname(iv),sct%vname(iv))
        enddo
      endif
      call end_define(ncid)
      start_recd=1
    endif
    call writenc(ncid,'lon',sct%x,start_recd)
    call writenc(ncid,'lat',sct%y,start_recd)
    call writenc(ncid,'time',sct%t,start_recd)
    do iv=1,sct%nv
      call writenc(ncid,sct%vname(iv),sct%v(:,iv),start_recd)
    enddo
    call close_nc(ncid)
  end subroutine outsct_nc
  
  subroutine outsct_nc_append(path,sct)
    character(len=*),intent(in) :: path
    type(sct2d_type),intent(in) :: sct
    integer :: ncid,iv,i,rec
    character(len=256) :: filename,patho
    character :: slash='/'
    character(len=14) :: ctime
    logical :: ext
    if(sct%n<1)return
    i=len_trim(path);if(path(i:i)==slash)slash=' '
    do i=1,sct%n
      ctime=datestr(sct%t(i))
      patho=trim(path)//trim(slash)//ctime(1:4)//'/'
      call system('mkdir -p '//trim(patho))
      filename=trim(patho)//trim(sct%cflag)//'.sct.'//ctime(1:8)//'.nc'
      inquire(file=filename,exist=ext)
      if(.not.ext)then
        call open_nc(ncid,filename,'c')
        call dimension_define(ncid,'recd',0)
        call variable_define(ncid,'time',nf_double,['recd'])
        call variable_define(ncid,'lon',nf_real,['recd'])
        call variable_define(ncid,'lat',nf_real,['recd'])
        do iv=1,sct%nv
          call variable_define(ncid,sct%vname(iv),nf_real,['recd'])
        enddo
        call end_define(ncid)
        rec=1
      else
        call open_nc(ncid,filename,'w')
        rec=get_dimension_len(ncid,'recd')+1
      endif
      call writenc(ncid,'lon',sct%x(i),rec)
      call writenc(ncid,'lat',sct%y(i),rec)
      call writenc(ncid,'time',sct%t(i),rec)
      do iv=1,sct%nv
        call writenc(ncid,sct%vname(iv),sct%v(i,iv),rec)
      enddo
      call close_nc(ncid)
    enddo
  end subroutine outsct_nc_append
  
!-------------------------------------------------------------------------------------------------
  subroutine readin_sctobs(outpath,cflag,ctime,limit,ivs,sct)
    character(len=*),intent(in) :: outpath,cflag,ctime
    integer,intent(in) :: ivs(:)
    real(4),intent(in) :: limit(4)
    type(sct2d_type),intent(inout) :: sct
    
    character(len=200) :: filename
    character(len=14) :: str
    real(4) :: x,y
    real(4),allocatable :: var(:)
    integer :: i,j,nclm,ioerr
    filename=trim(outpath)//trim(cflag)//'.'//ctime(1:8)//'.sct'
    open(101,file=trim(filename),status='old')
    sct%n=0
    do while (.true.)
      read(101,*,iostat=ioerr)str,x,y
      if(ioerr/=0)exit
      if(x>=limit(1) .and. x<=limit(2) .and. y>=limit(3) .and. y<=limit(4))then
        sct%n=sct%n+1
      endif
    enddo
    close(101)
    sct%nv=size(ivs);nclm=maxval(ivs);allocate(var(nclm))
    allocate(sct%t(sct%n),sct%x(sct%n),sct%y(sct%n),sct%v(sct%n,sct%nv))
    open(101,file=trim(filename),status='old')
    i=0
    do while (.true.)
      read(101,*,iostat=ioerr)str,x,y,var
      if(ioerr/=0)exit
      if(x>=limit(1) .and. x<=limit(2) .and. y>=limit(3) .and. y<=limit(4))then
        i=i+1
        sct%t(i)=datenum(str)
        sct%x(i)=x
        sct%y(i)=y
        do j=1,sct%nv
          sct%v(i,j)=var(ivs(j))
        enddo
      endif
    enddo
    close(101)
    deallocate(var)
    
  end subroutine readin_sctobs
  
  subroutine output_sctobs(outpath,cflag,ctime,sct)
    character(len=*),intent(in) :: outpath,cflag,ctime
    type(sct2d_type),intent(in) :: sct
    character(len=200) :: filename
    integer :: i
    character(len=50) :: strform
    filename=trim(outpath)//trim(cflag)//'.'//ctime(1:8)//'.sct'
    write(strform,"('(a14,',i2,'f12.4)')")sct%nv+2
    open(101,file=trim(filename),status='old')
    do i=1,sct%n
      !write(101,"(a14,<sct%nv+2>f12.4)")datestr(sct%t(i)),sct%x(i),sct%y(i),sct%v(i,:)
      write(101,strform)datestr(sct%t(i)),sct%x(i),sct%y(i),sct%v(i,:)
    enddo
    close(101)
  end subroutine output_sctobs
!-------------------------------------------------------------------------------------------------
!BOP
! !IROUTINE:  output_sctdata
! !DESCRIPTION: This routine does something to the input variable and returns
! the result in the output variable.
!\\\\
! !INTERFACE:
  subroutine output_sctdata(outpath,cflag,typ,ctime,lon,lat,&
                            var1,var2,var3,var4,var5,var6,var7,var8,var9,act)
! !INPUT PARAMETERS: 
    character(len=*),intent(in) :: outpath,cflag,ctime(:)
    integer,intent(in) :: typ(:)
    real(4),intent(in) :: lon(:),lat(:),var1(:)
    real(4),intent(in),optional :: var2(:),var3(:),var4(:),var5(:)
    real(4),intent(in),optional :: var6(:),var7(:),var8(:),var9(:)
    integer,intent(in),optional :: act
!EOP
!BOC
    integer :: nv,i
    character(len=256) :: filename
    filename=trim(outpath)//trim(cflag)//'.'//ctime(1)(1:8)//'.sct'
    nv=1
    if(present(var2))nv=2
    if(present(var3))nv=3
    if(present(var4))nv=4
    if(present(var5))nv=5
    if(present(var6))nv=6
    if(present(var7))nv=7
    if(present(var8))nv=8
    if(present(var9))nv=9
    !do i=1,size(lon)
    !  if(lon(i)<0)lon(i)=lon(i)+360
    !enddo
    if(present(act))then
      open(101,file=trim(filename),status='replace',form='formatted')
    else
      open(101,file=trim(filename),status='unknown',form='formatted')
    endif
    if(nv==1)then
      do i=1,size(lon)
        write(101,'(a15,3f15.5,i4)')ctime(i),lon(i),lat(i),var1(i),typ(i)
      enddo
    elseif(nv==2)then
      do i=1,size(lon)
        write(101,'(a15,4f15.5,i4)')ctime(i),lon(i),lat(i),var1(i),var2(i),typ(i)
      enddo
    elseif(nv==3)then
      do i=1,size(lon)
        write(101,'(a15,5f15.5,i4)')ctime(i),lon(i),lat(i),var1(i),var2(i),var3(i),typ(i)
      enddo
    elseif(nv==4)then
      do i=1,size(lon)
        write(101,'(a15,6f15.5,i4)')ctime(i),lon(i),lat(i),var1(i) &
                                ,var2(i),var3(i),var4(i),typ(i)
      enddo
    elseif(nv==5)then
      do i=1,size(lon)
        write(101,'(a15,7f15.5,i4)')ctime(i),lon(i),lat(i),var1(i) &
                                ,var2(i),var3(i),var4(i),var5(i),typ(i)
      enddo
    elseif(nv==6)then
      do i=1,size(lon)
        write(101,'(a15,8f15.5,i4)')ctime(i),lon(i),lat(i),var1(i) &
                                ,var2(i),var3(i),var4(i),var5(i),var6(i),typ(i)
      enddo
    elseif(nv==7)then
      do i=1,size(lon)
        write(101,'(a15,9f15.5,i4)')ctime(i),lon(i),lat(i),var1(i) &
                                 ,var2(i),var3(i),var4(i),var5(i),var6(i),var7(i),typ(i)
      enddo
    elseif(nv==8)then
      do i=1,size(lon)
        write(101,'(a15,10f15.5,i4)')ctime(i),lon(i),lat(i),var1(i) &
                                 ,var2(i),var3(i),var4(i),var5(i),var6(i),var7(i),var8(i),typ(i)
      enddo
    elseif(nv==9)then
      do i=1,size(lon)
        write(101,'(a15,11f15.5,i4)')ctime(i),lon(i),lat(i),var1(i) &
                                 ,var2(i),var3(i),var4(i),var5(i)   &
                                 ,var6(i),var7(i),var8(i),var9(i),typ(i)
      enddo
    endif
    close(101)
  end subroutine output_sctdata
!-------------------------------------------------------------------------------------------------
!BOP
! !IROUTINE: release_fld2d
! !DESCRIPTION:
!   To release the memory and pointer in data type of ocndata\_dat\_type
!\\\\
! !INTERFACE:
  subroutine release_fld2d(fld2d)
! !INPUT/OUTPUT PARAMETERS: 
    type(fld2d_var_type),intent(inout) :: fld2d
!EOP
!BOC
    if(associated(fld2d%lon))then;deallocate(fld2d%lon);nullify(fld2d%lon);endif
    if(associated(fld2d%lat))then;deallocate(fld2d%lat);nullify(fld2d%lat);endif
    if(associated(fld2d%var))then;deallocate(fld2d%var);nullify(fld2d%var);endif
    if(associated(fld2d%err))then;deallocate(fld2d%err);nullify(fld2d%err);endif
    fld2d%flag=0;fld2d%nodata=1;fld2d%im=0;fld2d%jm=0;
!EOC
  end subroutine release_fld2d
!-------------------------------------------------------------------------------------------------
!BOP
! !IROUTINE:  outfld2d
! !DESCRIPTION:
!  To release memory for type of field observations.
!\\\\
! !INTERFACE:
  subroutine outfld2d(fld2d,ctime,cflag,outpath)
    type(fld2d_var_type),intent(in) :: fld2d
    character(len=*),intent(in) :: ctime
    character(len=*),intent(in) :: cflag
    character(len=*),intent(in) :: outpath
!EOP
!BOC
    integer :: flag
    !4:lon,lat,var,err
    !3:lon,lat,var
    flag=3;if(associated(fld2d%err))flag=4
    call system('mkdir -p '//trim(outpath)//ctime(1:4))
    open(11,file=trim(outpath)//ctime(1:4)//'/'//trim(cflag)//'.'//ctime(1:8)//'.fld',form='unformatted')
    write(11)fld2d%im,fld2d%jm,fld2d%misv,flag
    write(11)fld2d%var
    if(flag==4)write(11)fld2d%err
    write(11)fld2d%lon,fld2d%lat
    close(11)
!EOC
  end subroutine outfld2d
!-------------------------------------------------------------------------------------------------
!BOP
! !IROUTINE:  outfld2d_nc
! !INTERFACE:
  subroutine outfld2d_nc(fld2d,outpath)
! !INPUT PARAMETERS:
    type(fld2d_var_type),intent(in) :: fld2d
    !character(len=*),intent(in) :: ctime
    !character(len=*),intent(in) :: cflag
    character(len=*),intent(in) :: outpath
! !DESCRIPTION:
!  To release memory for type of field observations.
!\\\\
!EOP
    integer :: ncid,nv !nv=4:lon,lat,var,err;3:lon,lat,var
    character(len=256) :: filename
    nv=3;if(associated(fld2d%err))nv=4
    call system('mkdir -p '//trim(outpath)//fld2d%ctime(1:4))
    filename=trim(outpath)//fld2d%ctime(1:4)//'/'//trim(fld2d%cflag)//'.' &
           //fld2d%ctime(1:8)//'.nc'
    write(*,*)trim(filename)
    call open_nc(ncid,trim(filename),'c')
    call dimension_define(ncid,'lon',fld2d%im,'lon',nf_real)
    call dimension_define(ncid,'lat',fld2d%jm,'lat',nf_real)
    call set_attribute(ncid,'units','degrees_north','lat')
    call set_attribute(ncid,'units','degrees_east','lon')
    call set_attribute(ncid,'modulo','','lon')
    call variable_define(ncid,fld2d%vname,nf_real,['lon','lat'])
    call set_attribute(ncid,'missing_value',fld2d%misv,fld2d%vname)
    if(nv==4)then
      call variable_define(ncid,trim(fld2d%vname)//'err',nf_real,['lon','lat'])
    endif
    call set_attribute(ncid,'varnumb',nv)
    call end_define(ncid)
    call writenc(ncid,'lon',fld2d%lon)
    call writenc(ncid,'lat',fld2d%lat)
    call writenc(ncid,trim(fld2d%vname),fld2d%var)
    if(nv==4)then
      call writenc(ncid,trim(fld2d%vname)//'err',fld2d%err)
    endif
    call close_nc(ncid)
  end subroutine outfld2d_nc

  subroutine inpfld2d_nc(fld2d,datpath,act,irec)
! !OUTPUT PARAMETERS: 
    type(fld2d_var_type),intent(inout) :: fld2d
! !INPUT PARAMETERS: 
    character(len=*),intent(in) :: datpath
    integer,optional,intent(in) :: act,irec
    integer :: ncid,recd,vndims
    logical :: ext
    character(len=256) :: filename
    real(4),allocatable :: vtmp(:,:,:)
    filename=trim(datpath)//fld2d%ctime(1:4)//'/'//trim(fld2d%cflag)//'.' &
           //fld2d%ctime(1:8)//'.nc'
    inquire(file=trim(filename),exist=ext)
    if(.not. ext)then
      filename=trim(datpath)//fld2d%ctime(1:4)//'/'//trim(fld2d%cflag)//'_' &
             //fld2d%ctime(1:8)//'.nc'
    endif
    inquire(file=trim(filename),exist=ext)
    if(.not. ext)then
      filename=trim(datpath)//'/'//trim(fld2d%cflag)//'.' &
             //fld2d%ctime(1:8)//'.nc'
    endif
    inquire(file=trim(filename),exist=ext)
    if(.not. ext)then
      filename=trim(datpath)//'/'//trim(fld2d%cflag)//'_' &
             //fld2d%ctime(1:8)//'.nc'
    endif
    inquire(file=trim(filename),exist=ext)
    if(.not. ext)then
      fld2d%nodata=1;fld2d%im=0;fld2d%jm=0;return
    else
      fld2d%nodata=0
    endif
    if(present(act))then
      if(act==0)return
      fld2d%im=1;fld2d%jm=1
      !allocate(fld2d%err(fld2d%im,fld2d%jm))
      allocate(fld2d%var(fld2d%im,fld2d%jm))
      allocate(fld2d%lon(fld2d%im),fld2d%lat(fld2d%jm))
      return
    endif
    recd=1;if(present(irec))recd=irec
    write(*,*)trim(filename)
    if(len_trim(fld2d%xname)==0)fld2d%xname='lon'
    if(len_trim(fld2d%yname)==0)fld2d%yname='lat'
    call open_nc(ncid,trim(filename),'r')
    fld2d%im=get_dimension_len(ncid,trim(fld2d%xname))
    fld2d%jm=get_dimension_len(ncid,trim(fld2d%yname))
    allocate(fld2d%lon(fld2d%im))
    allocate(fld2d%lat(fld2d%jm))
    allocate(fld2d%var(fld2d%im,fld2d%jm))
    call readnc(ncid,trim(fld2d%xname),fld2d%lon)
    call readnc(ncid,trim(fld2d%yname),fld2d%lat)
    vndims=get_varndims(ncid,trim(fld2d%vname))
    if(vndims>3)then
      allocate(vtmp(fld2d%im,fld2d%jm,1))
      call readnc(ncid,trim(fld2d%vname),vtmp,recd)
      fld2d%var=vtmp(:,:,1)
      deallocate(vtmp)
    else
      call readnc(ncid,trim(fld2d%vname),fld2d%var,recd)
    endif
    if(check_variable(ncid,trim(fld2d%vname)//'err'))then
      allocate(fld2d%err(fld2d%im,fld2d%jm))
      call readnc(ncid,trim(fld2d%vname)//'err',fld2d%err,recd)
    endif
    call close_nc(ncid)
  end subroutine inpfld2d_nc

  subroutine inpfld2d_nc_var(fld2d,datpath,act)
! !OUTPUT PARAMETERS: 
    type(fld2d_var_type),intent(inout) :: fld2d
! !INPUT PARAMETERS: 
    character(len=*),intent(in) :: datpath
    integer,optional,intent(in) :: act
    integer :: ncid
    logical :: ext
    character(len=256) :: filename
    filename=trim(datpath)//fld2d%ctime(1:4)//'/'//trim(fld2d%cflag)//'.' &
           //fld2d%ctime(1:8)//'.nc'
    write(*,*)trim(filename)
    inquire(file=trim(filename),exist=ext)
    if(.not. ext)then
      fld2d%nodata=1;fld2d%im=0;fld2d%jm=0;return
    else
      fld2d%nodata=0
    endif
    if(present(act))then
      if(act==0)return
      fld2d%im=1;fld2d%jm=1
      !allocate(fld2d%err(fld2d%im,fld2d%jm))
      allocate(fld2d%var(fld2d%im,fld2d%jm))
      allocate(fld2d%lon(fld2d%im),fld2d%lat(fld2d%jm))
      return
    endif
    call open_nc(ncid,trim(filename),'r')
    call readnc(ncid,trim(fld2d%vname),fld2d%var)
    if(check_variable(ncid,trim(fld2d%vname)//'err'))then
      if(.not.associated(fld2d%err))allocate(fld2d%err(fld2d%im,fld2d%jm))
      call readnc(ncid,trim(fld2d%vname)//'err',fld2d%var)
    endif
    call close_nc(ncid)
  end subroutine inpfld2d_nc_var
!-------------------------------------------------------------------------------------------------
!BOP
! !IROUTINE:  inpfld2d
! !INTERFACE:
  subroutine inpfld2d(datapath,cflag,ctime,fld2d,act)
! !INPUT PARAMETERS: 
    character(len=*),intent(in) :: datapath
    character(len=*),intent(in) :: cflag
    character(len=*),intent(in) :: ctime
    integer,intent(in),optional :: act
! !OUTPUT PARAMETERS: 
    type(fld2d_var_type),intent(out) :: fld2d
! !DESCRIPTION:
!  To release memory for type of field observations.
!\\\\
!EOP
!BOC
    integer :: flag
    logical :: ext
    character(len=256) :: filename
    !4:lon,lat,var,err
    !3:lon,lat,var
    filename=trim(datapath)//ctime(1:4)//'/'//trim(cflag)//'.'//ctime(1:8)//'.fld'
    inquire(file=trim(filename),exist=ext)
    if(.not. ext)then
      fld2d%nodata=1;fld2d%im=0;fld2d%jm=0;return
    else
      fld2d%nodata=0
    endif
    if(present(act))then
      if(act==0)return
      fld2d%im=1;fld2d%jm=1
      allocate(fld2d%err(fld2d%im,fld2d%jm))
      allocate(fld2d%var(fld2d%im,fld2d%jm),fld2d%lon(fld2d%im),fld2d%lat(fld2d%jm))
      return
    endif
    write(*,*)trim(filename)
    open(11,file=trim(filename),form='unformatted')
    read(11)fld2d%im,fld2d%jm,fld2d%misv,fld2d%flag
    allocate(fld2d%var(fld2d%im,fld2d%jm),fld2d%lon(fld2d%im),fld2d%lat(fld2d%jm))
    read(11)fld2d%var
    allocate(fld2d%err(fld2d%im,fld2d%jm))
    if(fld2d%flag==4)then
      allocate(fld2d%err(fld2d%im,fld2d%jm))
      read(11)fld2d%err
    else
      fld2d%err=0.0
    endif
    read(11)fld2d%lon,fld2d%lat
    close(11)
  end subroutine inpfld2d
!EOC
!-------------------------------------------------------------------------------------------------
!BOP
! !IROUTINE: release_ocndat
! !DESCRIPTION:
!  To release the memory and pointer in data type of ocndata\_dat\_type.
!\\\\
! !INTERFACE:
  subroutine release_ocndat(ocndata)
! !INPUT/OUTPUT PARAMETERS: 
    type(ocndata_dat_type),intent(inout) :: ocndata
!EOP
!BOC
    if(associated(ocndata%x    ))deallocate(ocndata%x    ); nullify(ocndata%x    )
    if(associated(ocndata%y    ))deallocate(ocndata%y    ); nullify(ocndata%y    )
    if(associated(ocndata%z    ))deallocate(ocndata%z    ); nullify(ocndata%z    )
    if(associated(ocndata%var  ))deallocate(ocndata%var  ); nullify(ocndata%var  )
    if(associated(ocndata%varid))deallocate(ocndata%varid); nullify(ocndata%varid)
    if(associated(ocndata%flag ))deallocate(ocndata%flag ); nullify(ocndata%flag )
  end subroutine release_ocndat
!EOC
!-------------------------------------------------------------------------------------------------
!BOP
! !IROUTINE: set_prof_flnm
! !DESCRIPTION:
!   Set filename use path, cflag and ctime. The format of filename: trim(cflag)//'\_'//ctime(1:8).
!   If flag is given, check the directory named by 4-digits of year. Once it is not exist, a new
!   directory will be created through system.
!\\\\
! !INTERFACE:
  subroutine set_prof_flnm(path,cflag,ctime,filename,flag)
! !INPUT PARAMETERS: 
    character(len=*),intent(in) :: path  ! directory for the data.
    character(len=*),intent(in) :: cflag ! a character to specify the data type.
    character(len=*),intent(in) :: ctime ! the time for the data in character 
                                         ! form: 'yyyymmddHHMMSS'.
    integer,intent(in),optional :: flag  ! for whether checking the directory.
! !OUTPUT PARAMETERS:
    character(len=*),intent(out) :: filename
                                         ! filename for the data, the path is contained.
!EOP
!BOC
    logical :: ext
    !if(present(flag))then
    !  inquire(file=trim(path)//ctime(1:4)//'/ok',exist=ext)
    !  !if(.not.ext)call system('mkdir '//trim(path)//ctime(1:4))
    !  if(.not.ext)call system('mkdir '//trim(path)//ctime(1:4))
    !  open(121,file=trim(path)//ctime(1:4)//'/ok');close(121)
    !endif
    call system('mkdir -p '//trim(path)//ctime(1:4))
    filename=trim(path)//ctime(1:4)//'/'//trim(cflag)//'_'//ctime(1:8)
  end subroutine set_prof_flnm
!-------------------------------------------------------------------------------------------------
! - Syntax
!   out_oneprof_data(path,cflag,prof)
!   * character(len=*) :: path=directory for the data.
!   * character(len=*) :: cflag=a character to specify the data type.
!   * type(ocndata_dat_type) :: prof=one prof data need to output.
! - Purpose
!-------------------------------------------------------------------------------------------------
!EOC
!-------------------------------------------------------------------------------------------------
!BOP
! !IROUTINE: out_oneprof_data
! !DESCRIPTION:
!   One profile data will be output into files. The filename will be determined by the time of
!   this profile. If the file exist, the profile will be appended at the end of this file. Before
!   the process of one data set, the directory need to be empty. The format of the file is binary.
!\\\\
! !INTERFACE:
  subroutine out_oneprof_data(path,cflag,prof)
! !INPUT PARAMETERS: 
    character(len=*),intent(in) :: path,cflag
    type(ocndata_dat_type),intent(in) :: prof
!EOP
!BOC
    character(len=200) :: filename
    logical :: ext
    character(len=14) :: ctime
    ctime=datestr(prof%time)
    call set_prof_flnm(path,cflag,ctime,filename,1)
    filename=trim(filename)//'.prof'
    inquire(file=filename,exist=ext)
    if(ext)then
      open(109,file=filename,form='unformatted',status='old',access='append')
    else
      open(109,file=filename,form='unformatted',status='new')
    endif
    write(109)prof%ksize,prof%nv,prof%x,prof%y,prof%time,prof%flag
    write(109)prof%varid
    write(109)prof%z
    write(109)prof%var
    close(109)
!EOC
  end subroutine out_oneprof_data
!-------------------------------------------------------------------------------------------------
! - Syntax
!   out_prof_data(prof,filename)
!   * type(ocndata_dat_type) :: prof(:)=a set of profiles to be output.
!   * character(len=*) :: filename=filename for output data.
!-------------------------------------------------------------------------------------------------
!BOP
! !IROUTINE: out_prof_data
! !DESCRIPTION:
!   A set of profiles will be output into file named by filename. If the file exist, the output
!   will replace it. So the prof contains all the data within one day. The format of the file is
!   binary.
!\\\\
! !INTERFACE:
  subroutine out_prof_data(prof,filename)
! !INPUT PARAMETERS: 
    type(ocndata_dat_type),intent(in) :: prof(:)
    character(len=*),intent(in) :: filename
!EOP
!BOC
    integer :: i
    open(109,file=trim(filename)//'.prof',form='unformatted',status='replace')
    do i=1,size(prof)
      write(109)prof(i)%ksize,prof(i)%nv,prof(i)%x,prof(i)%y,prof(i)%time,prof(i)%flag
      write(109)prof(i)%varid
      write(109)prof(i)%z
      write(109)prof(i)%var
    enddo
    close(109)
  end subroutine out_prof_data
!-------------------------------------------------------------------------------------------------
! - Syntax
!   out_prof_data_asci(prof,filename)
!   * type(ocndata_dat_type) :: prof(:)=a set of profiles to be output.
!   * character(len=*) :: filename=filename for output data.
! - Purpose
!   A set of profiles will be output into file named by filename. If the file exist, the output
!   will replace it. So the prof contains all the data within one day. The format of the file is
!   ASCII.
!-------------------------------------------------------------------------------------------------
  subroutine out_prof_data_asci(prof,filename)
    type(ocndata_dat_type),intent(in) :: prof(:)
    character(len=*),intent(in) :: filename
    integer :: i
    open(109,file=trim(filename)//'.prf',status='replace')
    do i=1,size(prof)
      write(109,*)prof(i)%ksize,prof(i)%nv,prof(i)%x,prof(i)%y,prof(i)%time,prof(i)%flag
      write(109,*)prof(i)%varid
      write(109,*)prof(i)%z
      write(109,*)prof(i)%var
    enddo
    close(109)
  end subroutine out_prof_data_asci
!-------------------------------------------------------------------------------------------------
! - Syntax
!   profnumb=get_prof_no(filename,limits)
!   * real(4) :: limits(4)=spatial constrains for required profiles. [xl,xu,yl,yu]
!   * character(len=*) :: filename=file need to be input.
!   $ integer :: profnumb=the number of profiles in filename, constrained by limits in spatial.
! - Purpose
!   Check the number of profiles before input the data. The limits in spatial is used to select
!   required profiles.The format of the input file is binary.
!-------------------------------------------------------------------------------------------------
  integer function get_prof_no(filename,limits)
    real(4),intent(in) :: limits(4)
    character(len=*),intent(in) :: filename
    logical :: ext
    real(4) :: lon,lat,a,b
    real(8) :: days
    integer :: i,idx,flag,n_levels,var_no,ids(2),ierr
    integer,dimension(:),allocatable :: var_id    ! (var_no)
    real(4),dimension(:,:),allocatable :: values  ! (n_levels,var_no)
    real(4),dimension(:),allocatable :: depth     ! (n_levels)
    get_prof_no=0
#ifdef OUTTXTPRF
    inquire(file=trim(filename)//'.prf',exist=ext)
    if(.not.ext)return
    open(109,file=trim(filename)//'.prf',status='old')
    do while(.true.)
      read(109,*,iostat=ierr)n_levels,var_no,lon,lat,days,ids
      if(ierr/=0)exit
      if(var_no<0 .or. var_no>100)exit
      !read(109,end=209)n_levels,var_no,lon,lat,days,ids
      allocate(var_id(var_no),values(n_levels,var_no),depth(n_levels))
      read(109,*)var_id
      read(109,*)depth
      read(109,*)values

      if (.not.(lon<limits(1) .or. lon>limits(2) &
                .or. lat<limits(3) .or. lat>limits(4)))then
        flag=1
        do idx=1,var_no
          a=minval(values(:,idx));b=maxval(values(:,idx))
          if(var_id(idx)==ocnobsvid_temp)then
            if(a<-5 .or. b>35)flag=-1
          endif
          if(var_id(idx)==ocnobsvid_salt)then
            if(a<0. .or. b>40)flag=-1
          endif
        enddo
        if(flag==1)get_prof_no=get_prof_no+1
      endif
      deallocate(var_id,values,depth)
    enddo
#else
    inquire(file=trim(filename)//'.prof',exist=ext)
    if(.not.ext)return
    open(109,file=trim(filename)//'.prof',form='unformatted',status='old')
    do while(.true.)
      read(109,iostat=ierr)n_levels,var_no,lon,lat,days,ids
      if(ierr/=0)exit
      if(var_no<0 .or. var_no>100)exit
      !read(109,end=209)n_levels,var_no,lon,lat,days,ids
      allocate(var_id(var_no),values(n_levels,var_no),depth(n_levels))
      read(109)var_id
      read(109)depth
      read(109)values

      if (.not.(lon<limits(1) .or. lon>limits(2) &
                .or. lat<limits(3) .or. lat>limits(4)))then
        flag=1
        do idx=1,var_no
          a=minval(values(:,idx));b=maxval(values(:,idx))
          if(var_id(idx)==ocnobsvid_temp)then
            if(a<-5 .or. b>35)flag=-1
          endif
          if(var_id(idx)==ocnobsvid_salt)then
            if(a<0. .or. b>40)flag=-1
          endif
        enddo
        if(flag==1)get_prof_no=get_prof_no+1
      endif
      deallocate(var_id,values,depth)
    enddo
#endif
    209 continue
    close(109)
    if(allocated(values))deallocate(var_id,values,depth)
  end function get_prof_no
!-------------------------------------------------------------------------------------------------
! - Syntax
!   get_prof_data(filename,limits,prof)
!   * real(4) :: limits(4)=spatial constrains for required profiles. [xl,xu,yl,yu]
!   * character(len=*) :: filename=file need to be input.
!   $ type(ocndata_dat_type) :: prof(:)=a set of profiles to be input.
! - Purpose
!   After checking the number of profiles limited in spatial, a set of profiles will be input as
!   prof. The format of the input file is binary.
!-------------------------------------------------------------------------------------------------
  subroutine get_prof_data(filename,limits,prof)
    real(4),intent(in) :: limits(4)
    character(len=*),intent(in) :: filename
    type(ocndata_dat_type),intent(out) :: prof(:)
    integer :: n_levels,var_no,ids(2),ierr
    real(4) :: lon,lat
    real(8) :: days
    integer,dimension(:),allocatable :: var_id    ! (var_no)
    real(4),dimension(:,:),allocatable :: values  ! (n_levels,var_no)
    real(4),dimension(:),allocatable :: depth     ! (n_levels)
    integer :: i,num,idx,flag
    logical :: ext
    real(4) :: a,b
#ifdef OUTTXTPRF
    inquire(file=trim(filename)//'.prf',exist=ext)
    num=0; if(.not.ext)return
    open(109,file=trim(filename)//'.prf',status='old')
    do while (.true.)
      !read(109,end=210)n_levels,var_no,lon,lat,days,ids
      read(109,*,iostat=ierr)n_levels,var_no,lon,lat,days,ids
      if(ierr/=0)exit
      if(var_no<0 .or. var_no>100)exit
      allocate(var_id(var_no),values(n_levels,var_no),depth(n_levels))
      read(109,*)var_id
      read(109,*)depth
      read(109,*)values
      write(*,"('pprff',6f12.5)")lon,lat,limits
      if (.not.(lon<limits(1) .or. lon>limits(2)  &
                .or. lat<limits(3) .or. lat>limits(4)))then
        flag=1
        do idx=1,var_no
          a=minval(values(:,idx))
          b=maxval(values(:,idx))
          if(var_id(idx)==ocnobsvid_temp)then
            if(a<-5 .or. b>35)flag=-1
          endif
          if(var_id(idx)==ocnobsvid_salt)then
            if(a<0. .or. b>40)flag=-1
          endif
        enddo
        if(flag==1)then
          num=num+1
          allocate(prof(num)%varid(var_no))
          allocate(prof(num)%var(1,1,n_levels,var_no))
          allocate(prof(num)%x(1),prof(num)%y(1 ))
          allocate(prof(num)%z(n_levels))
          allocate(prof(num)%flag(2))
          prof(num)%isize=1
          prof(num)%jsize=1
          prof(num)%ksize=n_levels
          prof(num)%nv=var_no
          prof(num)%x=lon
          prof(num)%y=lat
          prof(num)%z=-abs(depth)
          prof(num)%time=days
          prof(num)%flag=ids
          prof(num)%varid=var_id
          prof(num)%var(1,1,:,:)=values(:,:)
        endif
      endif
      deallocate(var_id,values,depth)
    enddo
#else
    inquire(file=trim(filename)//'.prof',exist=ext)
    num=0; if(.not.ext)return
    open(109,file=trim(filename)//'.prof',form='unformatted',status='old')
    do while (.true.)
      !read(109,end=210)n_levels,var_no,lon,lat,days,ids
      read(109,iostat=ierr)n_levels,var_no,lon,lat,days,ids
      if(ierr/=0)exit
      if(var_no<0 .or. var_no>100)exit
      allocate(var_id(var_no),values(n_levels,var_no),depth(n_levels))
      read(109)var_id
      read(109)depth
      read(109)values
      if (.not.(lon<limits(1) .or. lon>limits(2) .or. lat<limits(3) .or. lat>limits(4)))then
        flag=1
        do idx=1,var_no
          a=minval(values(:,idx))
          b=maxval(values(:,idx))
          if(var_id(idx)==ocnobsvid_temp)then
            if(a<-5 .or. b>35)flag=-1
          endif
          if(var_id(idx)==ocnobsvid_salt)then
            if(a<0. .or. b>40)flag=-1
          endif
        enddo
        if(flag==1)then
          num=num+1
          allocate(prof(num)%varid(var_no))
          allocate(prof(num)%var(1,1,n_levels,var_no))
          allocate(prof(num)%x(1),prof(num)%y(1 ))
          allocate(prof(num)%z(n_levels))
          allocate(prof(num)%flag(2))
          prof(num)%isize=1
          prof(num)%jsize=1
          prof(num)%ksize=n_levels
          prof(num)%nv=var_no
          prof(num)%x=lon
          prof(num)%y=lat
          prof(num)%z=-abs(depth)
          prof(num)%time=days
          prof(num)%flag=ids
          prof(num)%varid=var_id
          prof(num)%var(1,1,:,:)=values(:,:)
        endif
      endif
      deallocate(var_id,values,depth)
    enddo
#endif
    210 continue
    close(109)
    if(allocated(values))deallocate(var_id,values,depth)
  end subroutine get_prof_data
!-------------------------------------------------------------------------------------------------
! - Syntax
!   release_ocndata_profs([profs])
!   @ type(ocndata_profs_type) :: profs
! - Purpose
!   To release the memory and pointer in data type of ocndata_profs_type. If profs is given, the
!   profs will be released. Else, the fixedprofs defined in this module will be released.
!-------------------------------------------------------------------------------------------------
  subroutine release_ocndata_profs(profs)
    type(ocndata_profs_type),intent(inout),optional :: profs
    integer :: i
    if(present(profs))then
      do i=1,profs%n;call release_ocndat(profs%ocndat(i));enddo
      if(associated(profs%ocndat))deallocate(profs%ocndat)
      nullify(profs%ocndat);profs%n=0
    else
      do i=1,fixedprofs%n;call release_ocndat(fixedprofs%ocndat(i));enddo
      if(associated(fixedprofs%ocndat))deallocate(fixedprofs%ocndat)
      nullify(fixedprofs%ocndat);fixedprofs%n=0
    endif
  end subroutine release_ocndata_profs
!-------------------------------------------------------------------------------------------------
! - Syntax
!   input_profs(datapath,cflag,slimits,tlimits)
!   * character(len=*) :: datapath=directory for input data.
!   * character(len=*) :: cflag=a character to specify the data type.
!   * real(4) :: slimits(4)=spatial constrains for required profiles. [xl,xu,yl,yu]
!   * real(8) :: tlimits(2)=limits in temporal [tl,tu]
! - Purpose
!   All profiles limted in spatial (slimits) and temporal (tlimts) will be input. The fixedprofs
!   defined in this module will be used for these data sets.
!-------------------------------------------------------------------------------------------------
  subroutine input_profs(datapath,cflag,slimits,tlimits)
    character(len=*),intent(in) :: datapath,cflag
    real(4),intent(in) :: slimits(4)
    real(8),intent(in) :: tlimits(2)
    integer :: idx,j,k
    character(len=14) :: ctime
    real(8) :: dtime
    character(len=100) :: filename
    integer,allocatable :: numbs(:)
    real(4) :: dx,dy
    integer :: idtime,idays
    call release_ocndata_profs
    idays=tlimits(2)-tlimits(1)+1
    allocate(numbs(idays));numbs=0
    !do dtime=tlimits(1),tlimits(2),1.d0
    do idtime=1,idays  
      dtime=tlimits(1)+dble(idtime)-1.d0    
      ctime=datestr(dtime)
      call set_prof_flnm(datapath,cflag,ctime,filename)
      numbs(idtime)=get_prof_no(filename,slimits)
      write(*,*)trim(filename),numbs(idtime)
    enddo

    fixedprofs%n=sum(numbs)
    if(fixedprofs%n>0)then
      allocate(fixedprofs%ocndat(fixedprofs%n))
      j=0
      !do dtime=tlimits(1),tlimits(2),1.d0
      do idtime=1,idays  
        dtime=tlimits(1)+dble(idtime)-1.d0
        ctime=datestr(dtime)
        if(numbs(idtime)==0)cycle
        call set_prof_flnm(datapath,cflag,ctime,filename)
        call get_prof_data(filename,slimits,fixedprofs%ocndat(j+1:j+numbs(idtime)))
        j=j+numbs(idtime)
      enddo
    endif

    if(allocated(numbs))deallocate(numbs)
  end subroutine input_profs
!EOC
!-------------------------------------------------------------------------------------------------
  end module ocndata_mod
!-------------------------------------------------------------------------------------------------
!$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$
