!$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$
!-------------------------------------------------------------------------------

  module asmtl_eakf_mod

!-------------------------------------------------------------------------------
!                                                Copyright (C) 2013 Xunqiang Yin
!                                                MODULE NAME : asmtl_eakf_mod
!                                                PRESENT VERSION : 2013-10-20
!
! --- USAGE : assimilation tools of ensemble adjustment Kalman filter 
!             (Ref: Anderson 2001, 2003).
! --- DEPEND: None
!
! --- NOTE for describing of subroutine / function :
!  A. The parameters bracketed with [], means optional parameter.
!  B. The describe for the parameters of subroutine / function, started with:
!   * It means input prameter;
!   $ It means output prameter;
!   @ It means input and output prameter(it will be changed inside).
!
!-------------------------------------------------------------------------------
! ***                           INTERFACE DESCRIBE                           ***
!-------------------------------------------------------------------------------
!
!  1. subroutine asmtl_eakf_mod_init(ens_size_)
!     * integer :: ens_size_ = Size of ensemble.
!     - Initialize the module of eakf_mod. It will allocate the module arraries in this module.
!
!-------------------------------------------------------------------------------------------------
!
!  2. subroutine asmtl_eakf_mod_final
!     - Finalize the module of eakf_mod. It will release the module arraries in this module.
!
!-------------------------------------------------------------------------------------------------
!
!  3. subroutine asmtl_eakf_for_obsloc(obs, obs_var)
!     * real(8) :: obs, obs_var
!
!-------------------------------------------------------------------------------------------------
!
!  4. subroutine asmtl_eakf_for_modloc(omg)
!     * real(8) :: omg
!
!-------------------------------------------------------------------------------------------------
!
!  5. function asmtl_omega(a, d)
!     * real(8) :: a
!     * real(8) :: d
!     $ real(8) :: asmtl_omega
!
!-------------------------------------------------------------------------------------------------
!
!                                                                    --- Xunqiang Yin, 2013/10/20
!                                                                   E-Mail: XunqiangYin@gmail.com
!
!-------------------------------------------------------------------------------------------------

  implicit none

!-------------------------------------------------------------------------------------------------

  public :: asmtl_eakf_mod_init,asmtl_eakf_mod_final
  public :: asmtl_eakf_for_obsloc,asmtl_eakf_for_modloc,asmtl_omega,degen_lim
  public :: mobs,ens,newens,dyo,dyp,dxp !,eakfbuf
  public :: cp_rms,cp_corc
  public :: inflation_factor
  private

!-------------------------------------------------------------------------------------------------

  integer :: ens_size

  real(8), pointer :: mobs(:)
  real(8), pointer :: ens(:)
  real(8), pointer :: newens(:)
  real(8), pointer :: dyo(:)
  real(8), pointer :: dyp(:)
  real(8), pointer :: dxp(:)
!  real(8), pointer :: eakfbuf(:)
  real(8), parameter :: degen_lim = 1e-10
  real(8) :: inflation_factor=1.0

  interface asmtl_omega
    module procedure asmtl_omega_rd,asmtl_omega_rr,asmtl_omega_dd
  end interface asmtl_omega

!-------------------------------------------------------------------------------------------------
  contains
!-------------------------------------------------------------------------------------------------

  subroutine asmtl_eakf_mod_init(ens_size_)
    integer, intent(in) :: ens_size_
    ens_size = ens_size_
    allocate(dxp(ens_size))
    allocate(mobs(ens_size))
    allocate(ens(ens_size))
    allocate(newens(ens_size))
!    allocate(eakfbuf(ens_size*2))
!    dyo => eakfbuf(1:ens_size)
!    dyp => eakfbuf(ens_size+1:ens_size*2)
    allocate(dyo(ens_size))
    allocate(dyp(ens_size))
  end subroutine asmtl_eakf_mod_init

  subroutine asmtl_eakf_mod_final
    if(associated(mobs)  )deallocate(mobs)  ;nullify(mobs)
    if(associated(ens)   )deallocate(ens)   ;nullify(ens)
    if(associated(newens))deallocate(newens);nullify(newens)
    if(associated(dxp)   )deallocate(dxp)   ;nullify(dxp)
    if(associated(dyp)   )deallocate(dyp)   ;nullify(dyp)
    if(associated(dyo)   )deallocate(dyo)   ;nullify(dyo)
    !if(associated(eakfbuf))deallocate(eakfbuf); nullify(eakfbuf, dyo, dyp)
  end subroutine asmtl_eakf_mod_final

!-------------------------------------------------------------------------------

  subroutine asmtl_eakf_for_obsloc(obs,obs_var)
    real(8), intent(in) :: obs,obs_var
    real(8) :: ymean,xmean,ystd2,r2
    ymean=sum(mobs)/dble(ens_size);dyp=mobs-ymean
    ystd2=sum(dyp**2)/dble(ens_size)
    if(ystd2<degen_lim)then
      dyo=0.d0
    else
      r2=ystd2*inflation_factor/obs_var
      dyo=ymean/(1.+r2)+obs/(1.+1./r2)+dyp/sqrt(1.+r2)-mobs
      dyo=dyo/ystd2
    endif
  end subroutine asmtl_eakf_for_obsloc

  subroutine asmtl_eakf_for_modloc(omg)
    real(8), intent(in) :: omg
    ! input: ens, dyp, dyo
    ! dyp & dyo need to be prepared before this sub.
    real(8) :: xmean,cov
    if(omg<degen_lim)then
      newens=ens;return
    endif
    xmean=sum(ens)/dble(ens_size)
    dxp=ens-xmean
    cov=omg*sum(dxp*dyp)/dble(ens_size)
    !if(cov < degen_lim)then
    if(abs(cov)<degen_lim)then
      newens=ens
    else
      newens=ens+cov*dyo
    endif
  end subroutine asmtl_eakf_for_modloc

!-------------------------------------------------------------------------------

  real(8) function asmtl_omega_rd(a,d)
    real(4),intent(in) :: a
    real(8),intent(in) :: d
    real(8) :: b,ra,rd
    ra=a;rd=d
    asmtl_omega_rd=asmtl_omega(ra,rd)
  end function asmtl_omega_rd

  real(8) function asmtl_omega_rr(a,d)
    real(4),intent(in) :: a
    real(4),intent(in) :: d
    real(8) :: b,ra,rd
    ra=a;rd=d
    asmtl_omega_rr=asmtl_omega(ra,rd)
  end function asmtl_omega_rr

  real(8) function asmtl_omega_dd(a,d)
    real(8), intent(in) :: a, d
    real(8) :: b
    if(a==0)then
      asmtl_omega_dd=1.0;return
    endif
    b=abs(d)/a
    asmtl_omega_dd=0.0
    if(b<=1)then
      asmtl_omega_dd=-0.25d0    *b**5 &
                    +0.50d0     *b**4 &
                    +(5.d0/8.d0)*b**3 &
                    -(5.d0/3.d0)*b**2 &
                    +1.0
    elseif(b<=2)then
      asmtl_omega_dd=(1.d0/12.d0)*b**5 &
                    -0.50d0      *b**4 &
                    +(5.d0/8.)   *b**3 &
                    +(5.d0/3.)   *b**2 &
                    -5.d0        *b    &
                    +4.d0              &
                    -(2/3.)/b
    endif
    if(asmtl_omega_dd<0)asmtl_omega_dd=0.d0
  end function asmtl_omega_dd

!-------------------------------------------------------------------------------

  real(8) function cp_rms(x)
    real(8),intent(in) :: x(:)
    real(8) :: nn
    nn=size(x)
    cp_rms=sqrt(nn*sum(x*x)-sum(x)*sum(x))/nn
    !sum((x-sum(x)/nn)^2)=sum(x*x-2*(sum(x)/nn)*x+(sum(x)/nn)^2)
    !                    =sum(x*x)-sum(x)*sum(x)/nn
    !                    =(nn*sum(x*x)-sum(x)*sum(x))/nn
    !rms=sqrt(sum((x-sum(x)/nn)^2)/nn)
    !   =sqrt((nn*sum(x*x)-sum(x)*sum(x)))/nn
    !real(8) :: xb,nn
    !nn=size(x);xb=sum(x)/nn;cp_rms=sqrt(sum((x-xb)**2)/nn)
  end function cp_rms

  real(8) function cp_corc(x,y)
    real(8),intent(in) :: x(:),y(:)
    !---------------------------------------------------------------------------
    !  where, xb = sum(x)/nn; yb = sum(y)/nn => sum(x)=nn*xb; sum(y)=nn*yb
    !  nn*sum((x-xb)(y-yb)) = nn*sum(x*y-xb*y-x*yb-xb*yb) 
    !                       = nn*(sum(x*y)-xb*sum(y)-sum(x)*yb+nn*xb*yb)
    !                       = nn*(sum(x*y)-nn*xb*yb)
    !                       = nn*sum(x*y)-sum(x)*sum(y)
    !  corc = mean((x-xb)(y-yb))/sqrt(mean((x-xb)^2)*mean((y-yb)^2))
    !       = sum((x-xb)(y-yb))/sqrt(sum((x-xb)^2)*sum((y-yb)^2))
    !       = nn*sum((x-xb)(y-yb))/sqrt(nn*sum((x-xb)^2)*nn*sum((y-yb)^2))
    !       = [nn*sum(x*y)-sum(x)*sum(x)] / sqrt[ (nn*sum(x*x)-sum(x)*sum(x)) *
    !                                             (nn*sum(y*y)-sum(y)*sum(y)) ]
    !---------------------------------------------------------------------------
    real(8) :: nn,xb,yb,xyb,x2b,y2b,xvar,yvar
    real(8),parameter :: small=1.e-5
    nn=size(x);xb=sum(x);yb=sum(y)
    xyb=sum(x*y);x2b=sum(x*x);y2b=sum(y*y)
    xvar=nn*x2b-xb*xb
    yvar=nn*y2b-yb*yb  ! rms*rms*nn*nn
    if(xvar<small .or. yvar<small)then
      cp_corc=0
    else
      cp_corc=(nn*xyb-xb*yb)/sqrt(xvar*yvar)
    endif
  end function cp_corc
!-------------------------------------------------------------------------------

  end module asmtl_eakf_mod

!-------------------------------------------------------------------------------
!$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$$
