#define JXJ
!/ ------------------------------------------------------------------- /
!  f2py -c --fcompiler=gnu95 -m uom_st6 uom_st6.f90
!/ ------------------------------------------------------------------- /
      MODULE uomst6_mod
!
!/ ------------------------------------------------------------------- /
!     By Qingxiang Liu (UoM) on JAN-03-2018.
!
!     This module contains all the subroutines designed for the source
!     term package ST6, which was developed by the ocean engineering
!     group @ the Uni. of Melbourne (UoM). Please refer to the following
!     papers for further details of all these parameterizations:
!         Donelan et al.    (2006, JPO)   [Sin]
!         Young and Babanin (2006, JPO)   [Sds]
!         Tsagareli et al.  (2010, JPO)   [Sin]
!         Babanin et al.    (2010, JPO)   [Sds]
!         Rogers et al.     (2012, JTECH) [ST6 in SWAN]
!         Zieger et al.     (2015, OM)    [ST6 in WW3]
!         Liu et al.        (2017, OM)    [ST6 for hurricane cases]
!
!     Here we aimed to implement ST6 into the wave model MASNUM from the
!     first institute of oceanography, SOA (FIO). One detail should be
!     kept in mind that MASNUM uses the wavenumber spectrum E(k, θ) as
!     the basic spectrum, where wave height Hs = 4 √E(k, θ) k dk dθ.
!
!     In order to make sure our module consistent with the framework of
!     MASNUM, all the source terms here return the so-called growth rate
!     β, where δE/δt = β * E (i.e., derivative term in WW3) and the
!     source term S = β * E.
!
!/ ------------------------------------------------------------------- /
!/
      use mwmvar_mod,only: spdp

      IMPLICIT NONE
!
      PUBLIC  :: ST6SIN, ST6SDS, ST6SWL, ST6FLX
!
!     PRIVATE :: ST6FLX, LFACTOR, IRANGE, CALDSII, TAUWINDS
!
!/ ------------------------------------------------------------------- /
!/    XFR: the power of wavenumber bins (1.1**2)
!
      REAL(spdp), PARAMETER :: DAIR = 1.225, DWAT = 1000.,         &
                         GRAV = 9.806, TPI  = 6.2831854,     &
                         XFR  = 1.21, ONE = 1.0
!
#ifdef JXJ
      REAL(spdp), PARAMETER :: SIN6CF = 1.0,      &
                               SIN6WS = 32.,      &
                               SIN6A0 = 0.06,     &
                               SDS6A1 = 2.8E-6,   &
                               SDS6A2 = 3.5E-5,   &
                               SDS6P1 = 4.,       &
                               SDS6P2 = 4.,       &
                               SWL6B1 = 7.5E-3    ! MASNUM-original
#else
      REAL(spdp), PARAMETER :: SIN6CF = 1.0,       &
                               SIN6WS = 32.,       &
                               SIN6A0 = 0.09,      &
                               SDS6A1 = 4.75E-6,   &
                               SDS6A2 = 7.0E-5,    &
                               SDS6P1 = 4.,        &
                               SDS6P2 = 4.,        &
                               SWL6B1 = 4.1E-3     ! WW3 v6.07
#endif
!
      CONTAINS
!/ ------------------------------------------------------------------- /
      SUBROUTINE ST6FLX(CDFAC, U10, CD, UST)
!/ ------------------------------------------------------------------- /
!/
!/    Use the drag law suggested by Hwang (2011, JTECH) and later
!     modified by Rogers et al. (2012, JTECH) to calculate CD.
!
!/ ------------------------------------------------------------------- /
!     Input args:
!/ ------------------------------------------------------------------- /
!         CDFAC R. I  a parameter which can be used to increase/decrease
!                     CD so that we can partially cancel the bias in wind
!                     forcing. The default value will be 1.
!         U10   R. I  as its name suggest
!         CD    R. O  drag coefficient
!         UST   R. O  wind stress ustar
!/ ------------------------------------------------------------------- /
!/
      IMPLICIT NONE
!/
!/ ------------------------------------------------------------------- /
!/
      REAL(spdp), INTENT(IN)  :: CDFAC, U10
      REAL(spdp), INTENT(OUT) :: CD, UST
!/
!/ ------------------------------------------------------------------- /
!/
!     To prevent the drag coefficient from dropping to zero at extreme
!     wind speeds, we use a simple modification UST = 2.026 m/s for
!     U10 greater than 50.33 m/s.
!
      IF (U10 .GE. 50.33) THEN
          UST = 2.026 * SQRT(CDFAC)
          CD  = (UST/U10)**2
      ELSE
         CD  = CDFAC * ( 8.058 + 0.967*U10 - 0.016*U10**2 ) * 1E-4
         UST = U10 * SQRT(CD)
      END IF
!
      RETURN
!
      END SUBROUTINE ST6FLX
!
!/ ------------------------------------------------------------------- /
      SUBROUTINE ST6SIN (NK, NTH, DPT, U10, UDIR, WN, SIG, CG, THETA, &
                         E, S, D, mycd)

!/ ------------------------------------------------------------------- /
!     Wind input source term Sin
!/ ------------------------------------------------------------------- /
!/
!     Input args:
!/ ------------------------------------------------------------------- /
!      NK      I.   I  # of wavenumber bins
!      NTH     I.   I  # of direcitonal bins
!      DPT     R.   I  water depth
!      U10     R.   I  wind speed
!      UDIR    R.   I  wind direction
!      WN      R.A. I  wavenumber array
!      SIG     R.A. I  σ array
!      CG      R.A. I  group velocity
!      THETA   R.A. I  θ array
!      E       R.A. I  E(k, θ) (Hs = 4 √E(k, θ) k dk dθ)
!      S       R.A. O  Source term S = β * E
!      D       R.A. O  Derivative term β
!/
!/ ------------------------------------------------------------------- /
!/
      IMPLICIT NONE
!/
!/ ------------------------------------------------------------------- /
!/ Parameter list
      INTEGER, INTENT(IN)    :: NK, NTH
      REAL(spdp), INTENT(IN)       :: DPT, U10, UDIR
      REAL(spdp), INTENT(IN)       :: WN(NK), SIG(NK), CG(NK), THETA(NTH), &
                                E(NK, NTH)
      REAL(spdp), INTENT(OUT)      :: S(NK, NTH), D(NK, NTH)
      REAL(spdp), INTENT(out)      :: mycd  ! add by yinxq
!/
!/ ------------------------------------------------------------------- /
      INTEGER                :: ITH, IK
      REAL(spdp)                   :: DTH, COSU, SINU, CD10, USTAR, UPROXY
      REAL(spdp), DIMENSION(NK)    :: ADENSIG, FMAX, ANAR, SQRTBN, LFACT
      REAL(spdp), DIMENSION(NK,NTH):: ECOS2, ESIN2, SIG2, CINV2, CG2, WN2, &
                                F, SQRTBN2, W1, W2, SDENSIG

!/ ------------------------------------------------------------------- /
!/ 0) --- set up a basic variables ----------------------------------- /
!     cos(θu) & sin(θu)
      COSU   = COS(UDIR)
      SINU   = SIN(UDIR)
!
!     cos(θw) & sin(θw)
      DTH = TPI / NTH ! δθ
      DO ITH = 1, NTH
          ECOS2(1:NK, ITH) = COS(THETA(ITH))
          ESIN2(1:NK, ITH) = SIN(THETA(ITH))
      ENDDO
!
!     σ, 1/c
      DO IK = 1, NK
          WN2  (IK, 1:NTH) = WN(IK)             ! k
          CG2  (IK, 1:NTH) = CG(IK)             ! cg
          SIG2 (IK, 1:NTH) = SIG(IK)            ! σ
          CINV2(IK, 1:NTH) = WN(IK) / SIG(IK)   ! 1/c
      ENDDO
!
!/    --- scale friction velocity to wind speed (10m) in
!/        the boundary layer ----------------------------------------- /
!/    Donelan et al. (2006) used U10 or U_{λ/2} in their S_{in}
!/    parameterization. To avoid some disadvantages of using U10 or
!/    U_{λ/2}, Rogers et al. (2012) used the following engineering
!/    conversion:
!/                    UPROXY = SIN6WS * UST
!/
      CALL ST6FLX(SIN6CF, U10, CD10, USTAR)

      mycd=CD10

      UPROXY = SIN6WS * USTAR
!
!/ 1) --- calculate 1d wavenumber spectrum (F(k)) and
!/        zero-out values less than 1.0E-32 to avoid NaNs when
!/        computing directional narrowness in step 4). --------------- /
      DO IK = 1, NK
          F(IK, 1:NTH) = E(IK, 1:NTH) * WN(IK) ! F(k, θ) = E(k, θ) * k
      ENDDO
      ADENSIG = SUM(F, 2) * DTH ! ADENSIG: one dimensional F(k)
!
!/ 2) --- calculate normalised directional spectrum F(k, θ) ---------- /
      FMAX = MAXVAL(F, 2)
      DO IK = 1,NK
         IF (FMAX(IK).LT.1.0E-34) THEN
            F(IK, 1:NTH) = 1.
         ELSE
            F(IK, 1:NTH) = F(IK, 1:NTH) / FMAX(IK)
         END IF
      END DO
!
!/ 3) --- calculate normalised spectral saturation BN(IK) ------------ /
      ANAR    = 1.0/( SUM(F, 2) * DTH )         ! directional narrowness
!
      SQRTBN  = SQRT( ANAR * ADENSIG * WN**3 )
      DO IK  = 1, NK
          SQRTBN2(IK, 1:NTH) = SQRTBN(IK)
      END DO
!
!/ 4) --- calculate growth rate GAMMA and S for all directions for
!/        following winds (U10/c - 1 is positive; W1) and in 7) for
!/        adverse winds (U10/c -1 is negative, W2). W1 and W2
!/        complement one another. ------------------------------------ /
      W1      = MAX( 0., UPROXY * CINV2* ( ECOS2*COSU + ESIN2*SINU ) - 1. )**2
!
!     D -- β
      D       = (DAIR / DWAT) * SIG2 * &
                (2.8 - ( 1. + TANH(10.*SQRTBN2*W1 - 11.) )) *SQRTBN2*W1
!
      S       = D * E
!
!/ 5) --- calculate reduction factor LFACT using non-directional
!         spectral density of the wind input ------------------------- /
!     F(σ) = F(k) / Cg
      SDENSIG = S*WN2/CG2 ! S(σ)
      CALL LFACTOR(NK, NTH, DPT, U10, UDIR, USTAR, WN, SIG, THETA, &
                   SDENSIG, LFACT)
!
!/ 6) --- apply reduction (LFACT) to the entire spectrum ------------- /
      IF (SUM(LFACT) .LT. NK) THEN
          DO IK = 1, NK
              D(IK, 1:NTH) = D(IK, 1:NTH) * LFACT(IK)
          ENDDO
          S = D * E
      END IF
!
!/ 7) --- compute negative wind input for adverse winds. negative
!/        growth is typically smaller by a factor of ~2.5 (=.28/.11)
!/        than those for the favourable winds [Donelan, 2006, Eq. (7)].
!/        the factor is adjustable with NAMELIST parameter in
!/        ww3_grid.inp: '&SIN6 SINA0 = 0.04 /' ----------------------- /
      IF (SIN6A0.GT.0.0) THEN
        W2    = MIN( 0., UPROXY * CINV2* ( ECOS2*COSU + ESIN2*SINU ) - 1. )**2
        D     = D - ( (DAIR / DWAT) * SIG2 * SIN6A0 *                   &
                (2.8 - ( 1. + TANH(10.*SQRTBN2*W2 - 11.) )) *SQRTBN2*W2 )
        S     = D * E
      END IF
!
!/
!/ End of ST6SIN ----------------------------------------------------- /
!/
      END SUBROUTINE ST6SIN
!/ ------------------------------------------------------------------- /
      SUBROUTINE ST6SDS (NK, NTH, WN, SIG, CG, E, S, D)
!
!/ ------------------------------------------------------------------- /
!     wave-breaking term Sds
!/ ------------------------------------------------------------------- /
!
!     Input args:
!      NK      I.   I  # of wavenumber bins
!      NTH     I.   I  # of direcitonal bins
!      WN      R.A. I  wavenumber array
!      SIG     R.A. I  σ array
!      CG      R.A. I  group velocity
!      E       R.A. I  E(k, θ) (Hs = 4 √E(k, θ) k dk dθ)
!      S       R.A. O  Source term S = β * E
!      D       R.A. O  Derivative term β
!
!/ ------------------------------------------------------------------- /
!/
      IMPLICIT NONE
!/
!/ ------------------------------------------------------------------- /
!/ Parameter list
      INTEGER, INTENT(IN)  :: NK, NTH
      REAL(spdp), INTENT(IN)     :: WN(NK), SIG(NK), CG(NK), E(NK, NTH)
      REAL(spdp), INTENT(OUT)    :: S(NK, NTH), D(NK, NTH)
!/
!/ ------------------------------------------------------------------- /
!/ Local parameters
      INTEGER           :: IK
      REAL(spdp)              :: DTH
      REAL(spdp)              :: FREQ(NK)     ! frequencies [Hz]
      REAL(spdp)              :: DSII(NK)     ! δσ
      REAL(spdp)              :: DFII(NK)     ! frequency bandwiths [Hz]
      REAL(spdp)              :: ANAR(NK)     ! directional narrowness
      REAL(spdp)              :: BNT          ! empirical constant for
                                        ! wave breaking probability
      REAL(spdp)              :: EDENS (NK)   ! spectral density E(f)
      REAL(spdp)              :: ETDENS(NK)   ! threshold spec. density ET(f)
      REAL(spdp)              :: EXDENS(NK)   ! excess spectral density EX(f)
      REAL(spdp)              :: NEXDENS(NK)  ! normalised excess spectral density
      REAL(spdp)              :: T1(NK)       ! inherent breaking term
      REAL(spdp)              :: T2(NK)       ! forced dissipation term
      REAL(spdp)              :: T12(NK)      ! = T1+T2 or combined dissipation
      REAL(spdp)              :: ADF(NK)      ! temp. variables
!
!/ ------------------------------------------------------------------- /
!/ 0) --- Initialize essential parameters ---------------------------- /
      DTH     = TPI / NTH
      FREQ    = SIG/TPI
      DSII    = CALDSII(NK, SIG)
!
      ANAR    = 1.0
      BNT     = 0.035**2
      T1      = 0.0
      T2      = 0.0
      NEXDENS = 0.0
!
!/ 1) --- Calculate threshold spectral density, spectral density, and
!/        the level of exceedence EXDENS(f) -------------------------- /
      ETDENS  = ( TPI * BNT ) / ( ANAR * CG * WN**3 ) ! FT(f)
      EDENS   = SUM(E, 2) * DTH * WN * TPI / CG       ! F(f)
      EXDENS  = MAX(0.0,EDENS-ETDENS)
!
!/    --- normalise by a generic spectral density -------------------- /
      NEXDENS = EXDENS / ETDENS    ! normalise by threshold spectral density
!
!/ 2) --- Calculate inherent breaking component T1 ------------------- /
      T1 = SDS6A1 * ANAR * FREQ * (NEXDENS**SDS6P1)
!
!/ 3) --- Calculate T2, the dissipation of waves induced by
!/        the breaking of longer waves T2 ---------------------------- /
      ADF    = ANAR * (NEXDENS**SDS6P2)
      DO IK = 1,NK
         DFII = DSII/TPI
         IF (IK .GT. 1 .AND. IK .LT. NK) THEN
             DFII(IK) = 0.5 * (SIG(IK) - SIG(IK-1)) / TPI
         ENDIF
         T2(IK) = SDS6A2 * SUM( ADF(1:IK)*DFII(1:IK) )
      END DO
!
!/ 4) --- Sum up dissipation terms and apply to all directions ------- /
      T12 = -1.0 * ( MAX(0.0,T1)+MAX(0.0,T2) )
      DO IK = 1, NK
         D(IK, 1:NTH) = T12(IK)
      END DO
!
      S = D * E
!
!/
      END SUBROUTINE ST6SDS
!/ ------------------------------------------------------------------- /
      SUBROUTINE ST6SWL(NK, NTH, WN, SIG, CG, E, S, D)
!/ ------------------------------------------------------------------- /
!      Input args:
!/ ------------------------------------------------------------------- /
!/
!      NK      I.   I  # of wavenumber bins
!      NTH     I.   I  # of direcitonal bins
!      WN      R.A. I  wavenumber array
!      SIG     R.A. I  σ array
!      CG      R.A. I  group velocity
!      E       R.A. I  E(k, θ) (Hs = 4 √E(k, θ) k dk dθ)
!      S       R.A. O  Source term S = β * E
!      D       R.A. O  Derivative term β
!
!/ ------------------------------------------------------------------- /
      IMPLICIT NONE
!/
!/ ------------------------------------------------------------------- /
      INTEGER, INTENT(IN) :: NK, NTH
      REAL(spdp), INTENT(IN)    :: WN(NK), SIG(NK), CG(NK), E(NK, NTH)
      REAL(spdp), INTENT(OUT)   :: S(NK, NTH), D(NK, NTH)
!/
!/ ------------------------------------------------------------------- /
      INTEGER             :: IK
      REAL(spdp), DIMENSION(NK) :: DSII, FBAND, FMAX, ANAR, BN
      REAL(spdp)                :: DTH, F(NK, NTH), B1
!/
!/ ------------------------------------------------------------------- /
!/ 0) --- Initialize parameters -------------------------------------- /
      DTH   = TPI / NTH
      DSII  = CALDSII(NK, SIG)
      D     = 0.
!
!/ 1) --- Choose calculation of steepness a*k ------------------------ /
!/        Replace the measure of steepness with the spectral
!         saturation after Banner et al. (2002) ---------------------- /
      DO IK = 1, NK
          F(IK, 1:NTH) = E(IK, 1:NTH) * WN(IK) ! F(k, θ) = E(k, θ) * k
      ENDDO
      FBAND = SUM(F, 2) * DTH ! F(k)
!
      FMAX = MAXVAL(F, 2)
      DO IK = 1, NK
         IF (FMAX(IK) .LT. 1.0E-34) THEN
            F(IK, 1:NTH) = 1.
         ELSE
            F(IK, 1:NTH) = F(IK, 1:NTH) / FMAX(IK)
         END IF
      END DO
      ANAR  = 1.0/(SUM(F, 2) * DTH)
      BN    = ANAR * FBAND * WN**3
!
!/    --- A constant value for B1 attenuates swell too strong in the
!/        western central Pacific (i.e. cross swell less than 1.0m).
!/        Workaround is to scale B1 with steepness a*kp, where kp is
!/        the peak wavenumber. SWL6B1 remains a scaling constant, but
!/        with different magnitude.  --------------------------------- /
      IK    = MAXLOC(FBAND,1)         ! Index for peak
      B1    = SWL6B1 * ( 2. * SQRT(SUM(FBAND/CG*DSII)) * WN(IK) )
!
!/ 2) --- Calculate the derivative term only (in units of 1/s) ------- /
      DO IK = 1,NK
         IF (FBAND(IK) .GT. 1.E-30) THEN
            D(IK, 1:NTH) = -(2./3.) * B1 * SIG(IK) * SQRT(BN(IK))
         END IF
      END DO
!
!/ 3) --- Apply dissipation term of derivative to all directions ----- /
      S = D * E
!
      END SUBROUTINE ST6SWL
!/ ------------------------------------------------------------------- /
!/
      SUBROUTINE LFACTOR(NK, NTH, DPT, U10, UDIR, USTAR, WN, SIG, &
                         THETA, S, LFACT)
!/ ------------------------------------------------------------------- /
!     Reduce the wind input Sin dynamically so that the integrated
!     wave-supported stress plus the viscous stress are .LE. than
!     the total wind stress.
!/ ------------------------------------------------------------------- /
!     Input args:
!/ ------------------------------------------------------------------- /
!      NK      I.   I  # of wavenumber bins
!      NTH     I.   I  # of direcitonal bins
!      DPT     R    I  Water depth
!      U10     Real I  Wind speed (10m)
!      UDIR    Real I  Wind direction
!      USTAR   Real I  Friction velocity
!      WN      R.A. I  Wavenumber array
!      THETA   R.A. I  θ array
!      SIG     R.A. I  Relative frequencies [in rad.]
!      S       R.A. I  Wind input energy density spectrum  (S_{in}(σ, θ))
!      LFACT   R.A. O  Factor array                        LFACT(σ)
!     ----------------------------------------------------------------
      IMPLICIT NONE
!
!/ ------ I/O parameters --------------------------------------------- /
      INTEGER, INTENT(IN):: NK, NTH
      REAL(spdp), INTENT(IN)   :: DPT
      REAL(spdp), INTENT(IN)   :: U10, UDIR     ! wind speed
      REAL(spdp), INTENT(IN)   :: USTAR         ! friction velocity & direction
      REAL(spdp), INTENT(IN)   :: WN(NK)        ! wavenumber
      REAL(spdp), INTENT(IN)   :: SIG(NK)       ! relative frequencies
      REAL(spdp), INTENT(IN)   :: THETA(NTH)    ! relative frequencies
      REAL(spdp), INTENT(IN)   :: S(NK, NTH)    ! wind-input source term Sin
      REAL(spdp), INTENT(OUT)  :: LFACT(NK)     ! correction factor
!
!/    --- local parameters (in order of appearance) ------------------ /
      REAL(spdp), PARAMETER   :: FRQMAX  = 10.  ! Upper freq. limit to extrapolate to.
      INTEGER, PARAMETER:: ITERMAX = 80   ! Maximum number of iterations to
                                          ! find numerical solution for LFACT.
!
      INTEGER           :: IK, ITH, NK10Hz
      REAL(spdp)              :: WNMAX, DTH, ECOS2(NK, NTH), ESIN2(NK, NTH)
!
      REAL(spdp), ALLOCATABLE :: IK10Hz(:), WN10Hz(:), SIG10Hz(:), &
                           CINV10Hz(:), DSII10Hz(:), UCINV10Hz(:)
      REAL(spdp), ALLOCATABLE :: SDENS10Hz(:), SDENSX10Hz(:), SDENSY10Hz(:), &
                           LF10Hz(:)
      REAL(spdp)              :: TAU_TOT, TAU, TAU_VIS
      REAL(spdp)              :: TAUVX, TAUVY, TAUWX, TAUWY, TAUX, TAUY
      REAL(spdp)              :: TAU_INIT(2)
      REAL(spdp)              :: UPROXY, RTAU, DRTAU, ERR
      INTEGER           :: SIGN_NEW, SIGN_OLD
      LOGICAL           :: OVERSHOT
!
!/ ------------------------------------------------------------------- /
!
!/ 0) --- Find the number of frequencies required to extend arrays
!/        up to f=10Hz and allocate arrays --------------------------- /
!     Get WNMAX from FRQMAX by using the deep-water dispersion relation
!     ALOG is the same as LOG
      WNMAX  = (TPI * FRQMAX) ** 2. / GRAV
      !NK10Hz = CEILING(ALOG(WNMAX/WN(1))/ALOG(XFR))+1
      NK10Hz = CEILING(ALOG(real(WNMAX/WN(1)))/ALOG(real(XFR)))+1
      NK10Hz = MAX(NK,NK10Hz)
!
!     COS & SIN
      DTH = TPI / NTH ! δθ
      DO ITH = 1, NTH
          ECOS2(1:NK, ITH) = COS(THETA(ITH))
          ESIN2(1:NK, ITH) = SIN(THETA(ITH))
      ENDDO
!
      ALLOCATE(IK10Hz(NK10Hz))          ! index array
      IK10Hz = REAL( IRANGE(1,NK10Hz,1) )
!
      ALLOCATE(WN10Hz(NK10Hz))
      ALLOCATE(SIG10Hz(NK10Hz))
      ALLOCATE(DSII10Hz(NK10Hz))
      ALLOCATE(CINV10Hz(NK10Hz))
      ALLOCATE(UCINV10Hz(NK10Hz))
!
      ALLOCATE(SDENS10Hz(NK10Hz))
      ALLOCATE(SDENSX10Hz(NK10Hz))
      ALLOCATE(SDENSY10Hz(NK10Hz))
      ALLOCATE(LF10Hz(NK10Hz))
!
!/ 1) --- Either extrapolate arrays up to 10Hz or use discrete spectral
!         grid per se. Limit the constraint to the positive part of the
!         wind input only. ---------------------------------------------- /
      IF (NK .LT. NK10Hz) THEN
         SDENS10Hz(1:NK)         = SUM(S,2) * DTH                ! Sin(σ)
         SDENSX10Hz(1:NK)        = SUM(MAX(0.,S)*ECOS2, 2) * DTH ! Sin(σ)cos(θ)
         SDENSY10Hz(1:NK)        = SUM(MAX(0.,S)*ESIN2, 2) * DTH ! Sin(σ)sin(θ)
         WN10Hz                  = WN(1) * XFR ** (IK10Hz-1.0)
         SIG10Hz                 = SQRT(GRAV * WN10Hz * TANH(WN10Hz * DPT))
         CINV10Hz                = WN10Hz / SIG10Hz              ! 1/C
         DSII10Hz                = CALDSII(NK10Hz, SIG10Hz)
!
!        --- Spectral slope for S_IN(F) is proportional to F**(-2) ------ /
         SDENS10Hz(NK+1:NK10Hz)  = SDENS10Hz(NK)  * (SIG10Hz(NK)/SIG10Hz(NK+1:NK10Hz))**2
         SDENSX10Hz(NK+1:NK10Hz) = SDENSX10Hz(NK) * (SIG10Hz(NK)/SIG10Hz(NK+1:NK10Hz))**2
         SDENSY10hz(NK+1:NK10Hz) = SDENSY10Hz(NK) * (SIG10Hz(NK)/SIG10Hz(NK+1:NK10Hz))**2
      ELSE
         SIG10Hz                 = SIG
         CINV10Hz                = WN / SIG
         DSII10Hz                = CALDSII(NK, SIG)
         SDENS10Hz(1:NK)         = SUM(S,2) * DTH
         SDENSX10Hz(1:NK)        = SUM(MAX(0.,S)*ECOS2, 2) * DTH
         SDENSY10Hz(1:NK)        = SUM(MAX(0.,S)*ESIN2, 2) * DTH
      END IF
!
!/ 2) --- Stress calculation ----------------------------------------- /
!     Total stress (CD from Hwang (2011))
      TAU_TOT  = USTAR**2 * DAIR
!
!     Cv from Tsagareli et al. 2010
      TAU_VIS  = MAX(0.0, -5.0E-5*U10 + 1.1E-3) * U10**2 * DAIR
!     TAU_VIS  = MIN(0.9 * TAU_TOT, TAU_VIS)
      TAU_VIS  = MIN(0.95 * TAU_TOT, TAU_VIS)
!
      TAUVX    = TAU_VIS * COS(UDIR)
      TAUVY    = TAU_VIS * SIN(UDIR)
!
!     --- The wave supported stress. --------------------------------- /
      TAUWX    = TAUWINDS(SDENSX10Hz,CINV10Hz,DSII10Hz)   ! normal stress (x-component)
      TAUWY    = TAUWINDS(SDENSY10Hz,CINV10Hz,DSII10Hz)   ! normal stress (y-component)
      TAU_INIT = (/TAUWX,TAUWY/)                          ! unadjusted normal stress components
!
      TAUX     = TAUVX + TAUWX                            ! total stress (x-component)
      TAUY     = TAUVY + TAUWY                            ! total stress (y-component)
      TAU      = SQRT(TAUX**2  + TAUY**2)                 ! total stress (magnitude)
      ERR      = (TAU-TAU_TOT)/TAU_TOT                    ! initial error
!
!/ 3) --- Find reduced Sin(f) = L(f)*Sin(f) to satisfy our constraint
!/        TAU <= TAU_TOT --------------------------------------------- /
      LF10Hz = 1.0
      IK     = 0
!
      IF (TAU .GT. TAU_TOT) THEN
         OVERSHOT    = .FALSE.
         RTAU        = ERR / 90.
         DRTAU       = 2.0
         SIGN_NEW    = INT(SIGN(ONE,ERR))

         UPROXY      = SIN6WS * USTAR
         UCINV10Hz   = 1.0 - (UPROXY * CINV10Hz)
!
         DO IK=1,ITERMAX
            LF10Hz   = MIN(1.0, EXP(UCINV10Hz * RTAU) )
!
            TAUWX    = TAUWINDS(SDENSX10Hz*LF10Hz,CINV10Hz,DSII10Hz)
            TAUWY    = TAUWINDS(SDENSY10Hz*LF10Hz,CINV10Hz,DSII10Hz)
!
            TAUX     = TAUVX + TAUWX
            TAUY     = TAUVY + TAUWY
            TAU      = SQRT(TAUX**2 + TAUY**2)
            ERR      = (TAU-TAU_TOT) / TAU_TOT
!
            SIGN_OLD = SIGN_NEW
            SIGN_NEW = INT(SIGN(ONE, ERR))
!
!        --- Slow down DRTAU when overshot. -------------------------- /
            IF (SIGN_NEW .NE. SIGN_OLD) OVERSHOT = .TRUE.
            IF (OVERSHOT) DRTAU = MAX(0.5*(1.0+DRTAU),1.00010)
!
            RTAU = RTAU * (DRTAU**SIGN_NEW)
!
            IF (ABS(ERR) .LT. 1.54E-4) EXIT
         END DO
!
         ! IF (IK .GE. ITERMAX) WRITE (*, 280) U10, TAU, &
         !               TAU_TOT, ERR, TAUWX, TAUWY, TAUVX, TAUVY
      END IF
!
      LFACT(1:NK) = LF10Hz(1:NK)
!
      280 FORMAT ('WARNING LFACTOR (U10,TAU,TAU_TOT,ERR,TAUW_XY,'  &
                  'TAUV_XY): ',F6.1,2F7.4,E10.3,4F7.4)
!
      DEALLOCATE(IK10Hz,WN10Hz,SIG10Hz,CINV10Hz,DSII10Hz,LF10Hz)
      DEALLOCATE(SDENS10Hz,SDENSX10Hz,SDENSY10Hz,UCINV10Hz)
!/
      END SUBROUTINE LFACTOR
!/ ------------------------------------------------------------------- /
!/
      FUNCTION IRANGE(X0,X1,DX) RESULT(IX)
!
      IMPLICIT NONE
!
      INTEGER, INTENT(IN)  :: X0, X1, DX
      INTEGER, ALLOCATABLE :: IX(:)
      INTEGER              :: N
      INTEGER              :: I
!
      N = INT(REAL(X1-X0)/REAL(DX))+1
      ALLOCATE(IX(N))
      DO I = 1, N
         IX(I) = X0+ (I-1)*DX
      END DO
!/
      END FUNCTION IRANGE
!/
!/ ------------------------------------------------------------------- /
!/
      FUNCTION CALDSII(NK, SIG) RESULT(DSII)
!
!     Calculate dσ from σ array
!
      IMPLICIT NONE
!
      INTEGER, INTENT(IN) :: NK
      REAL(spdp), INTENT(IN)    :: SIG(NK)
      REAL(spdp)                :: DSII(NK)
!
      INTEGER             :: IK
!
      DSII = 0.
      DO IK = 1, NK
          IF (IK .EQ. 1) THEN
              DSII(IK) = (SIG(2) - SIG(1)) * 0.5
          ELSE IF (IK .EQ. NK) THEN
              DSII(IK) = (SIG(NK) - SIG(NK-1)) * 0.5
          ELSE
              DSII(IK) = 0.5 * (SIG(IK+1) - SIG(IK-1))
          ENDIF
      ENDDO
!
      END FUNCTION CALDSII
!/ ------------------------------------------------------------------- /
!/
      FUNCTION TAUWINDS(SDENSIG,CINV,DSII) RESULT(TAU_WINDS)
!
      IMPLICIT NONE
      REAL(spdp), INTENT(IN)  :: SDENSIG(:)    ! Sin(sigma) in [m2/rad-Hz]
      REAL(spdp), INTENT(IN)  :: CINV(:)       ! inverse phase speed
      REAL(spdp), INTENT(IN)  :: DSII(:)       ! freq. bandwidths in [radians]
      REAL(spdp)              :: TAU_WINDS     ! wind stress
!
      TAU_WINDS = GRAV * DWAT * SUM(SDENSIG*CINV*DSII)
!/
      END FUNCTION TAUWINDS
!/ ------------------------------------------------------------------- /
!/
!/ End of module W3SRC6MD -------------------------------------------- /
!/
      END MODULE uomst6_mod
