!-------------------------------------------------------------------------------
#define DBG print*,__FILE__,__LINE__
! #define MULTI_SOURCEFUNCTION
!BOP
! !MODULE: mwmsub_mod
! !INTERFACE:
  module mwmsub_mod
! !DESCRIPTION: This module contains subroutines needed for non-breaking wave
! mixing incorperated into circulation models. The results of bv could be added
! to moment diffusivity and tracer diffusivity calculated from those schemes
! without considering the wave effects. The information of grid/mesh including
! the longitude, lagitude, ocean depth, depth of each level and dimension values
! of the mesh in the space of wave number (or frequency, or circular frequency)
! and wave direction will be input from initial routine. And the wave spectrum
! will calculated using the same code of MASNUM wave model.
!\\
!\\
!  References:\\
!  * Qiao F, Yuan Y, Ezer T, et al.
!  A three-dimensional surface wave-ocean circulation coupled model and its initial testing.
!  Ocean Dynamics, 2010.
!\\
!\\
! !REVISION HISTORY:
!  2018/05/07 - Xunqiang Yin - Initial Version
!
! !REMARKS:
!
!  MWMSUB-2018
!
! !USES:

  use mwmvar_mod, only: spdp,np,npc,kl,jl,im,jm,deltt,delttm,grdflag,small
  use mwmvar_mod, only: nsp,pnb,cnb,d,alon,alat,ee,ea,wx,wy,h1_3,aet,ape,tpf,e
  use mwmvar_mod, only: wamvar_mod_final,wamvar_mod_init, cnb,deg2rad,pi,rs
!... added by BaoYing, 20190103
  use mwmvar_mod, only: ustokes,vstokes
!... end add 	

  use mwmcor_mod, only: setwave,implsch,nlweight,setspec
  use mwmcor_mod, only: check_timestep,intact_mixture_bv

  use mwmpgt_mod, only: propagat,init_propagat,final_propagat
  use mwmpgt_mod, only: cal_geo_distangl,setgeoinf,smoth_ee_dir
  implicit none

! !PUBLIC MEMBER FUNCTIONS:

  public :: mwmsub_init              ! - Initialize the surface wave model.
  public :: mwmsub_forward           ! - Forward wave model, curvilinear grid.
  public :: mwmsub_getvar            ! - Get variable from this module.
  public :: mwmsub_getsize           ! - Get grid size in wave number and direction.
  public :: mwmsub_final             ! - Finalize the wave model.
  private

  !-----------------------------------------------------------------------------!
  !                             *** INTERFACES ***                              !
  !-----------------------------------------------------------------------------!

  interface mwmsub_init
    module procedure &
            mwmsub_init_cuv,        &! - For normal partition of rectangle.
            mwmsub_init_blk          ! - For partition of multi-blocks.
  end interface mwmsub_init

  interface mwmsub_forward
    module procedure &
            mwmsub_forward_cuv,     &! - Forward wave model, partition by rectangle.
            mwmsub_forward_blk       ! - Forward wave model, partition by blocks.
  end interface mwmsub_forward

  interface mwmsub_getvar
    module procedure &
            mwmsub_getvar_byname_2d,&! - Get 2D var by name and return error flag.
            mwmsub_getvar_byname_3d,&! - Get 3D var by name and return error flag.
            mwmsub_getvar_byname_4d,&! - Get 4D var by name and return error flag.
            mwmsub_getvar_cuv      ,&! - Get variable from this module by a rectangle.
            mwmsub_getvar_blk        ! - Get variable from this module by blocks.
  end interface mwmsub_getvar

!EOP
!-------------------------------------------------------------------------------

  public :: mwmsub_forward_noprgt

  real(spdp),allocatable :: wx0(:),wy0(:),wx1(:),wy1(:)
  real(spdp),allocatable :: h3(:,:),bv0(:,:)
  integer,allocatable    :: nsp_init(:),np_halo(:)

  integer :: nx          = 0
  integer :: ny          = 0
  integer :: kb          = 0
  integer :: nblock      = 0
  integer :: total_step  = 0
  integer :: isfirstwind = 1
  integer :: iscoolstart = 1
  integer :: ineedvecrot = 0
  integer :: wind_type   = 0
  integer :: halo_size   = 1
  real(8) :: count_steps = 0

!-------------------------------------------------------------------------------
  contains
!-------------------------------------------------------------------------------
!BOP
! !IROUTINE: mwmsub_init
! !INTERFACE:
  subroutine mwmsub_init_cuv(tlon,tlat,mask,&! Grid locations and mask.
                      depth,zlevel,  &! Topography and vertical layers.
                      nstep,delttime,&! Steps and time interval of winds.
                      ksize,jsize,   &! Model size in physical space.
                      vecdir,        &! Vector direction, default is along grid.
                      wndtype,       &! Forcing type: 0-wind vector,1-windstress.
                      istart,        &! Type of initial condition.
                      halosize,      &! Halo size of this block.
                      inrbonds,      &! Boundary index of this block.
                      windx,windy    )! Initial wind vector 10m above surface.
! !INPUT PARAMETERS:
    real(8),intent(in) :: tlon(:,:)              !(nx,ny)
    ! --- Longitude at each grid, in degree east (0-360).
    real(8),intent(in) :: tlat(:,:)              !(nx,ny)
    ! --- Latitude at each grid, in degree north (-90-90).
    integer,intent(in) :: mask(:,:)              !(nx,ny)
    ! --- Mask at each grid, the value will be 0~2.
    !  0  land point.
    !  1  water points, but not at open boundary
    !  2  water point at open boundary.
    real(8),intent(in) :: depth(:,:)             !(nx,ny)
    ! --- Depth at each grid, positive value in meter.
    real(8),intent(in),optional :: zlevel(:)     !(kb)
    ! --- Depth of each level in vertical, positive value in meter.
! !INPUT/OUTPUT PARAMETERS:
    integer,intent(inout) :: nstep
    ! --- Step number of wave model within the time interval of wind.
    !     If it is not proper, the suggested value will be returned.
    !     This could be used to test if this module is initiated succesfully.
    !     If (input nstep) /= (output nstep), the initial of MWMSUB is failed.
! !INPUT PARAMETERS:
    real(8),intent(in) :: delttime
    ! --- Time interval of wind forcing, in seconds.
    integer,intent(in),optional :: ksize,jsize
    ! --- The grid size in physical space (wavenumber,wavedirection).
    integer,intent(in),optional :: vecdir
    ! --- Type of vector direction.
    !  0  Default, same direction with the grid system.
    !  1  direction of normal east/north.
    integer,intent(in),optional :: wndtype
    ! --- Forcing type/flag:
    !  0  Default, wind vectors at 10m above the surface.
    !  1  Windstress will be given from outside. (under develop)
    integer,intent(in),optional :: istart
    ! --- Type of initial condition.
    !  0  Default, it will be cool start.
    !  1  The wave model is continue run from a restart.
    integer,intent(in),optional :: halosize
    ! --- Size of halo region of this box.
    integer,intent(in),optional :: inrbonds(4)
    ! --- Local index in this box to indicate the inner points.
    real(8),intent(in),optional :: windx(:,:)    !(nx,ny)
    real(8),intent(in),optional :: windy(:,:)    !(nx,ny)
    ! --- Wind vectors on the time of inquiring the mixing coefficients (bv).
    !     The unit is m/s and they are at 10m above the sea surface with the
    !     same grid with the initialization.

! !DESCRIPTION:
!   This routine will be used to initialize this module. The step number
!   of the wave model within the time interval of wind forcing is required.
!   The horizontal/vertical grid information are also inquired. The horizontal
!   grid information includes the longitude, latitude, mask and depth in each grid.
!   The Vertical grid information includes the depth of each layer. The grid sizes
!   in geo-space will inquired by checking the arrary size. The other parameters
!   are optional and default value will be used if they are not given.
!\\\\
!EOP
    !---------------------------------------------------------------------
    real(8),allocatable ::  tlon_(:,:,:)
    real(8),allocatable ::  tlat_(:,:,:)
    real(8),allocatable :: depth_(:,:,:)
    integer,allocatable ::  mask_(:,:,:)
    real(8),allocatable :: windx_(:,:,:)
    real(8),allocatable :: windy_(:,:,:)
    !---------------------------------------------------------------------
    nx=size(mask,1)
    ny=size(mask,2)
    nblock=1
    allocate( tlon_(nx,ny,1)); tlon_(:,:,1)= tlon(:,:)
    allocate( tlat_(nx,ny,1)); tlat_(:,:,1)= tlat(:,:)
    allocate(depth_(nx,ny,1));depth_(:,:,1)=depth(:,:)
    allocate( mask_(nx,ny,1)); mask_(:,:,1)= mask(:,:)
    !---------------------------------------------------------------------
    if(present(windx))then
      allocate(windx_(nx,ny,1));windx_(:,:,1)=windx(:,:)
      allocate(windy_(nx,ny,1));windy_(:,:,1)=windy(:,:)
      call mwmsub_init_blk(tlon_,tlat_,mask_,&! Grid locations and mask.
                        depth_,zlevel,       &! Topography and vertical layers.
                        nstep,delttime,      &! Steps and time interval of winds.
                        ksize,jsize,         &! Model size in physical space.
                        vecdir,              &! Vector direction, default is along grid.
                        wndtype,             &! Forcing type: 0-wind vector,1-windstress.
                        istart,              &! Type of initial condition.
                        halosize,            &! Halo size of this block.
                        inrbonds,            &! Boundary index of this block.
                        windx_,windy_        )! Initial wind vector 10m above surface.
    !---------------------------------------------------------------------
    else
      call mwmsub_init_blk(tlon_,tlat_,mask_,&! Grid locations and mask.
                        depth_,zlevel,       &! Topography and vertical layers.
                        nstep,delttime,      &! Steps and time interval of winds.
                        ksize,jsize,         &! Model size in physical space.
                        vecdir,              &! Vector direction, default is along grid.
                        wndtype,             &! Forcing type: 0-wind vector,1-windstress.
                        istart,              &! Type of initial condition.
                        halosize,            &! Halo size of this block.
                        inrbonds             )! Boundary index of this block.
    endif
    if(allocated( tlon_))deallocate( tlon_)
    if(allocated( tlat_))deallocate( tlat_)
    if(allocated(depth_))deallocate(depth_)
    if(allocated( mask_))deallocate( mask_)
    if(allocated(windx_))deallocate(windx_)
    if(allocated(windy_))deallocate(windy_)
  end subroutine mwmsub_init_cuv
!-------------------------------------------------------------------------------
!BOP
! !INTERFACE:
  subroutine mwmsub_init_blk(tlon,tlat,mask,&! Grid locations and mask.
                      depth,zlevel,  &! Topography and vertical layers.
                      nstep,delttime,&! Steps and time interval of winds.
                      ksize,jsize,   &! Model size in physical space.
                      vecdir,        &! Vector direction, default is along grid.
                      wndtype,       &! Forcing type: 0-wind vector,1-windstress.
                      istart,        &! Type of initial condition.
                      halosize,      &! Halo size of this block.
                      inrbonds,      &! Boundary index of this block.
                      windx,windy    )! Initial wind vector 10m above surface.
! !INPUT PARAMETERS:
    real(8),intent(in) :: tlon(:,:,:)            ! (nx,ny,nblock)
    ! --- Longitude at each grid, in degree east (0-360).
    real(8),intent(in) :: tlat(:,:,:)            ! (nx,ny,nblock)
    ! --- Latitude at each grid, in degree north (-90-90).
    integer,intent(in) :: mask(:,:,:)            ! (nx,ny,nblock)
    ! --- Mask at each grid, the value will be 0~2.
    !  0  land point.
    !  1  water points, but not at open boundary
    !  2  water point at open boundary.
    real(8),intent(in) :: depth(:,:,:)           ! (nx,ny,nblock)
    ! --- Depth at each grid, positive value in meter.
    real(8),intent(in),optional :: zlevel(:)     ! (kb)
    ! --- Depth of each level in vertical.
! !INPUT/OUTPUT PARAMETERS:
    integer,intent(inout) :: nstep
    ! --- Step number of wave model within the time interval of wind.
    !     If it is not proper, the suggested value will be returned.
    !     This could be used to test if this module is initiated succesfully.
    !     If (input nstep) /= (output nstep), the initial of MWMSUB is failed.
! !INPUT PARAMETERS:
    real(8),intent(in) :: delttime
    ! --- Time interval of wind forcing, in seconds.
    integer,intent(in),optional :: ksize,jsize
    ! --- The grid size in physical space (wavenumber,wavedirection).
    integer,intent(in),optional :: vecdir
    ! --- Type of vector direction.
    !  0  Default, same direction with the grid system.
    !  1  direction of normal east/north.
    integer,intent(in),optional :: wndtype
    ! --- Forcing type/flag:
    !  0  Default, wind vectors at 10m above the surface.
    !  1  Windstress will be given from outside. (under develop)
    integer,intent(in),optional :: istart
    ! --- Type of initial condition.
    !  0  Default, it will be cool start.
    !  1  The wave model is continue run from a restart.
    integer,intent(in),optional :: halosize
    ! --- Size of halo region of this box.
    integer,intent(in),optional :: inrbonds(4)
    ! --- Local index in this box to indicate the inner points.
    real(8),intent(in),optional :: windx(:,:,:)  !(nx,ny,nblock)
    real(8),intent(in),optional :: windy(:,:,:)  !(nx,ny,nblock)
    ! --- Wind vectors on the time of inquiring the mixing coefficients (bv).
    !     The unit is m/s and they are at 10m above the sea surface with the
    !     same grid with the initialization.

! !DESCRIPTION:
!   This routine will be used to initialize this module. The step number
!   of the wave model within the time interval of wind forcing is required.
!   The horizontal/vertical grid information are also inquired. The horizontal
!   grid information includes the longitude, latitude, mask and depth in each grid.
!   The Vertical grid information includes the depth of each layer. The grid sizes
!   in geo-space will inquired by checking the arrary size. The other parameters
!   are optional and default value will be used if they are not given.
!\\\\
!EOP
    integer :: idx,flag,ksizetest,jsizetest,k
    !---------------------------------------------------------------------------
    ! --- 1. Set model sizes and allocate variables.
    !---------------------------------------------------------------------------
    call mwmsub_final
    if(present(ksize))kl=ksize
    if(present(jsize))jl=jsize
    !---------------------------------------------------------------------------
    ! --- 2. Initial model parameters related to grid and topography.
    !---------------------------------------------------------------------------
    !***************************************************************************
    call set_model_size_and_var(tlon,tlat,mask,depth,halosize,inrbonds)
    !***************************************************************************
    !---------------------------------------------------------------------------
    ! --- 3. Check time steps
    !---------------------------------------------------------------------------
    call setwave
    call nlweight
    total_step=nstep
    delttm=delttime/(dble(halo_size)*dble(nstep*60))
    call check_timestep(flag)
    if(flag==0)then
      nstep=ceiling(delttime/(deltt*halo_size))
      write(*,*)delttime,delttm,deltt
      return     ! ****** nstep is too small, please increase it.
    endif
    !---------------------------------------------------------------------------
    ! --- 4. Initial propagation and other parameters
    !---------------------------------------------------------------------------
    npc=np_halo(halo_size);call init_propagat;npc=np_halo(1)
    !------------------------------------------
    kb=size(zlevel)
    allocate(wx0(0:np),wy0(0:np),wx1(0:np),wy1(0:np),h3(kb,0:np),bv0(kb,0:np))
    do k=1,kb
      h3(k,:)=zlevel(k)
    enddo
    if(.not. allocated(ea))allocate(ea(kl,jl,0:np))
    count_steps=0.d0;ea=0.d0
    !------------------------------------------
    ineedvecrot=1  ! check vector directions.
    if(present(vecdir))then
      if(vecdir==1)ineedvecrot=0
    endif
    !------------------------------------------
    if(present(wndtype))wind_type=wndtype
    !------------------------------------------
    iscoolstart=1
    if(present(istart))then
      if(istart/=0)iscoolstart=0
    endif
    !---------------------------------------------------------------------------
    isfirstwind=1
    if(present(windx) .and. present(windy))then
      call gridtrans_2d_blk(windx,wx)
      call gridtrans_2d_blk(windy,wy)
      if(ineedvecrot==1)then
        do idx=1,np
          wx1(idx)=wx(idx)*cos(cnb(idx)%angle)-wy(idx)*sin(cnb(idx)%angle)
          wy1(idx)=wy(idx)*cos(cnb(idx)%angle)+wx(idx)*sin(cnb(idx)%angle)
        enddo
      else
        wx1=wx;wy1=wy
      endif
      wx0=wx1;wy0=wy1;isfirstwind=0
    endif
    !---------------------------------------------------------------------------
  end subroutine mwmsub_init_blk
!-------------------------------------------------------------------------------
!BOP
! !IROUTINE: mwmsub_forward
! !INTERFACE:
  subroutine mwmsub_forward_cuv(istep,   & ! Step index between 2 wind fields.
                            spectrum,    & ! Wave energy spectrum.
                            windx,windy, & ! Wind vector 10m above surface.
                            newmask,     & ! Ice mask for each grid, 0 or 1.
                            uvel,vvel    ) ! Circulation currents, m/s.
! !INPUT PARAMETERS:
    integer,intent(in) :: istep
    ! --- Index of steps within twice inquire of the mixing coefficients (bv)
    !     for coupling of the wave-circulation coupling. If 0 is given for it,
    !     there will be no wind interpolation along time and the wind provided
    !     into this subroutine will be used directly.
    real(8),intent(in) :: windx(:,:)              !(nx,ny)
    real(8),intent(in) :: windy(:,:)              !(nx,ny)
    ! --- Wind vectors on the time of inquiring the mixing coefficients (bv).
    !     The unit is m/s and they are at 10m above the sea surface with the
    !     same grid with the initialization.
    real(8),intent(in),optional :: newmask(:,:)   !(nx,ny)
    ! --- Mask fir ice coverage, 0 for land/ice grid and 1 for water grid.
    real(8),intent(in),optional :: uvel(:,:)      !(nx,ny)
    real(8),intent(in),optional :: vvel(:,:)      !(nx,ny)
    ! --- Background circulation (vertical averaged or at surface),
    !     the unit is m/s. (under develop)
! !INPUT/OUTPUT PARAMETERS:
    real(8),intent(inout) :: spectrum(:,:,:,:)    !(nx,ny,kl,jl)
    ! --- Energy spectrum of surface wave. This will need to readin from restart file
    !     or output for restart the model.
!
! !DESCRIPTION:
!   This routine will forward one/two step of the wave model. The wind vector
!   will be given, but the ice mask and current velocity are optional. The vector
!   could be agree with north/east (longitude/latitude) direction along or along
!   with the grid lines. The option of windstreees accepted in the updatedversion.
!\\\\
!EOP
    !---------------------------------------------------------------------------
    call gridtrans_4d(spectrum,ee)        ! --- Transfer spectrum to model grid.
    if(iscoolstart==1)then                ! New add by yinxq,20190121
      call set_wind_data;return           ! --- Set wind data
    endif                                 ! End of New add by yinxq,20190121
    call set_wind_data                    ! --- Set wind data
    call set_new_mask(newmask)            ! --- Set ice mask
    !---------------------------------------------------------------------------
    call forwardwave(istep)
    call gridtrans_4d_inv(ee,spectrum)    ! --- Transfer spectrum to model grid.
    !---------------------------------------------------------------------------
    contains
    !---------------------------------------------------------------------------
    subroutine set_wind_data
      integer :: idx
      !----------------------------------------
      if(istep==total_step .or. istep==0)then
        wx0=wx1;wy0=wy1
      endif
      call gridtrans_2d(windx,wx)
      call gridtrans_2d(windy,wy)
      if(ineedvecrot==1)then
        do idx=1,np
          wx1(idx)=wx(idx)*cos(cnb(idx)%angle)-wy(idx)*sin(cnb(idx)%angle)
          wy1(idx)=wy(idx)*cos(cnb(idx)%angle)+wx(idx)*sin(cnb(idx)%angle)
        enddo
      else
        wx1=wx;wy1=wy
      endif
      if(isfirstwind==1)then
        wx0=wx1;wy0=wy1;isfirstwind=0
      endif
      !----------------------------------------
      ! --- Set for cool start
      if(iscoolstart==1)then
        wx=wx0;wy=wy0
        do idx=1,np
          call setspec(1,idx)
        enddo
        iscoolstart=0
      endif
      !----------------------------------------
      !windweight=dble(istep)/dble(total_step)
      !if(istep/=0)windweight=1.d0
      !wx=windweight*wx1+(1.d0-windweight)*wx0
      !wy=windweight*wy1+(1.d0-windweight)*wy0
    end subroutine set_wind_data
    !---------------------------------------------------------------------------
    subroutine set_new_mask(newmask)
      real(8),intent(in),optional :: newmask(:,:)   !(nx,ny)
      integer :: idx
      if(present(newmask))then
        do idx=1,npc
          nsp(idx)=nsp_init(idx)*newmask(pnb(idx)%i,pnb(idx)%j)
        enddo
      endif
    end subroutine set_new_mask
  end subroutine mwmsub_forward_cuv
!-------------------------------------------------------------------------------
!BOP
! !INTERFACE:
  subroutine mwmsub_forward_blk(istep,      &!Step index between 2 wind fields.
                                spectrum,   &!Wave energy spectrum.
                                windx,windy,&!Wind vector 10m above surface.
                                newmask,    &!Ice mask for each grid, 0 or 1.
                                uvel,vvel   )!Circulation currents, m/s.
! !INPUT PARAMETERS:
    integer,intent(in) :: istep
    ! --- Index of steps within twice inquire of the mixing coefficients (bv)
    !     for coupling of the wave-circulation coupling. If 0 is given for it,
    !     there will be no wind interpolation along time and the wind provided
    !     into this subroutine will be used directly.
    real(8),intent(in) :: windx(:,:,:)              !(nx,ny)
    real(8),intent(in) :: windy(:,:,:)              !(nx,ny)
    ! --- Wind vectors on the time of inquiring the mixing coefficients (bv).
    !     The unit is m/s and they are at 10m above the sea surface with the
    !     same grid with the initialization.
    real(8),intent(in),optional :: newmask(:,:,:)   !(nx,ny)
    ! --- Mask fir ice coverage, 0 for land/ice grid and 1 for water grid.
    real(8),intent(in),optional :: uvel(:,:,:)      !(nx,ny)
    real(8),intent(in),optional :: vvel(:,:,:)      !(nx,ny)
    ! --- Background circulation (vertical averaged or at surface),
    !     the unit is m/s.
! !INPUT/OUTPUT PARAMETERS:
    real(8),intent(inout) :: spectrum(:,:,:,:,:)    !(nx,ny,kl,jl)
    ! --- Energy spectrum of surface wave. This will need to readin from restart file
    !     or output for restart the model.
!
! !DESCRIPTION:
!   This routine will forward one/two step of the wave model. The wind vector
!   will be given, but the ice mask and current velocity are optional. The vector
!   could be agree with north/east (longitude/latitude) direction along or along
!   with the grid lines. The option of windstreees accepted in the updatedversion.
!\\\\
!EOP
    !---------------------------------------------------------------------------
    call gridtrans_4d_blk(spectrum,ee)    ! --- Transfer spectrum to model grid.
    if(iscoolstart==1)then                ! New add by yinxq,20190121
      call set_wind_data;return           ! --- Set wind data
    endif                                 ! End of New add by yinxq,20190121
    call set_wind_data                    ! --- Set wind data
    call set_new_mask(newmask)            ! --- Set ice mask
    !---------------------------------------------------------------------------
    call forwardwave(istep)
    call gridtrans_4d_inv_blk(ee,spectrum) ! --- Transfer spectrum to model grid.
    !---------------------------------------------------------------------------
    contains
    !---------------------------------------------------------------------------
    subroutine set_wind_data
      integer :: idx
      !----------------------------------------
      if(istep==total_step .or. istep==0)then
        wx0=wx1;wy0=wy1
      endif
      call gridtrans_2d_blk(windx,wx)
      call gridtrans_2d_blk(windy,wy)
      if(ineedvecrot==1)then
        do idx=1,np
          wx1(idx)=wx(idx)*cos(cnb(idx)%angle)-wy(idx)*sin(cnb(idx)%angle)
          wy1(idx)=wy(idx)*cos(cnb(idx)%angle)+wx(idx)*sin(cnb(idx)%angle)
        enddo
      else
        wx1=wx;wy1=wy
      endif
      if(isfirstwind==1)then
        wx0=wx1;wy0=wy1;isfirstwind=0
      endif
      !----------------------------------------
      ! --- Set for cool start
      if(iscoolstart==1)then
        wx=wx0;wy=wy0
        do idx=1,np
          call setspec(1,idx)
        enddo
        iscoolstart=0
      endif
      !----------------------------------------
      !windweight=dble(istep)/dble(total_step)
      !if(istep/=0)windweight=1.d0
      !wx=windweight*wx1+(1.d0-windweight)*wx0
      !wy=windweight*wy1+(1.d0-windweight)*wy0
    end subroutine set_wind_data
    !---------------------------------------------------------------------------
    subroutine set_new_mask(newmask)
      real(8),intent(in),optional :: newmask(:,:,:)   !(nx,ny,nblock)
      integer :: idx
      if(present(newmask))then
        do idx=1,np
          nsp(idx)=nsp_init(idx)*newmask(pnb(idx)%i,pnb(idx)%j,pnb(idx)%k)
        enddo
      endif
    end subroutine set_new_mask
    !---------------------------------------------------------------------------
  end subroutine mwmsub_forward_blk
!-------------------------------------------------------------------------------
  subroutine forwardwave(istep)
    integer,intent(in) :: istep
    real(8) :: windweight
    integer :: ihalo,idx
    real(8) :: deltt_bk
#ifndef MULTI_SOURCEFUNCTION      
    deltt_bk=deltt;deltt=deltt_bk*total_step*halo_size
#endif      
    do ihalo=halo_size,1,-1
      !-------------------------------------------------------
      ee(:,:,0)=small;e(:,:,0)=small
      do idx=1,np_halo(ihalo) !npc
        call propagat(idx)                ! --- For wave propagation.
      enddo
      !-------------------------------------------------------
#ifndef MULTI_SOURCEFUNCTION      
      if(istep/=1 .or. ihalo/=halo_size)then
        ee=e;cycle
      endif
      windweight=1.d0
#else      
      windweight=dble((istep-1)*halo_size+ihalo-1) &
                /dble(total_step*halo_size)
      if(istep/=0)windweight=1.d0         ! 0: wind will be controled outside.
#endif      
      wx=windweight*wx1+(1.d0-windweight)*wx0
      wy=windweight*wy1+(1.d0-windweight)*wy0
      !-------------------------------------------------------
      do idx=1,np_halo(ihalo) !npc
        call implsch(idx)                 ! --- Source functions.
      enddo
      !call smoth_ee                      ! --- Smooth the wave spectrum.
#ifdef USEDIRSMOTH
      call smoth_ee_dir
#endif
      !-------------------------------------------------------
      ea=ea+ee;count_steps=count_steps+1.d0  ! --- Forward wave model
      !-------------------------------------------------------
    enddo
#ifndef MULTI_SOURCEFUNCTION      
    deltt=deltt_bk
#endif
  end subroutine forwardwave

  subroutine forwardwave1(istep)
    integer,intent(in) :: istep
    real(8) :: windweight
    integer :: ihalo,idx
    do ihalo=halo_size,1,-1
      !-------------------------------------------------------
      do idx=1,np_halo(ihalo) !npc
        call propagat(idx)                ! --- For wave propagation.
      enddo
      !-------------------------------------------------------
      if(istep/=1 .or. ihalo/=halo_size)cycle
      windweight=1.d0
      !-------------------------------------------------------
      !windweight=dble((istep-1)*halo_size+ihalo-1) &
      !          /dble(total_step*halo_size)
      !if(istep/=0)windweight=1.d0         ! 0: wind will be controled outside.
      wx=windweight*wx1+(1.d0-windweight)*wx0
      wy=windweight*wy1+(1.d0-windweight)*wy0
      !-------------------------------------------------------
      do idx=1,np_halo(ihalo) !npc
        call implsch(idx)                 ! --- Source functions.
      enddo
      !call smoth_ee                      ! --- Smooth the wave spectrum.
#ifdef USEDIRSMOTH
      call smoth_ee_dir
#endif
      !-------------------------------------------------------
      ea=ea+ee;count_steps=count_steps+1.d0  ! --- Forward wave model
      !-------------------------------------------------------
    enddo
  end subroutine forwardwave1
!-------------------------------------------------------------------------------
!BOP
! !IROUTINE: mwmsub_getvar
! !INTERFACE:
  subroutine mwmsub_getvar_byname_2d(var,vname,ierr)
! !INPUT PARAMETERS:
    character(len=*),intent(in) :: vname
! !OUTPUT PARAMETERS:
    real(8),intent(out) :: var(:,:)
    integer,intent(out) :: ierr
! !DESCRIPTION:
!   Get variables according to its name. For 2D variables, the name could be
!   "hs', "th", "tp" or "tz". For 3D variables, the name could be "bv", "hs',
!   "th", "tp" or "tz". For 4D variables, the name could be only "bv". The
!   meaning of "bv", "hs', "th", "tp" or "tz" are "Mixing coefficients induced
!   by non-breaking wave", "Significant wave height", "Mean wave direction",
!   "Zero-crossing wave period", and "Spectrum peak wave period" respectively.
!   This name is case-insensitive but must not contain blank space characters.
!\\\\
!EOP
    ierr=0;var=0
    select case (vname)
    case ('HS','Hs','hS','hs')
      call mwmsub_getvar_cuv(hs=var)
    case ('TH','Th','tH','th')
      call mwmsub_getvar_cuv(th=var)
    case ('TP','Tp','tP','tp')
      call mwmsub_getvar_cuv(tp=var)
    case ('TZ','Tz','tZ','tz')
      call mwmsub_getvar_cuv(tz=var)
!... added by BaoYing, 20190103
    case ('US','Us','uS','us')
      call mwmsub_getvar_cuv(us=var)
    case ('VS','Vs','vS','vs')
      call mwmsub_getvar_cuv(vs=var)
!... end add 	
    case default
      ierr=1
    end select
  end subroutine mwmsub_getvar_byname_2d
!BOP
! !INTERFACE:
  subroutine mwmsub_getvar_byname_3d(var,vname,ierr)
    character(len=*),intent(in) :: vname
! !OUTPUT PARAMETERS:
    real(8),intent(out) :: var(:,:,:)
    integer,intent(out) :: ierr
! !DESCRIPTION:
!   Get variables according to its name. For 2D variables, the name could be
!   "hs', "th", "tp" or "tz". For 3D variables, the name could be "bv", "hs',
!   "th", "tp" or "tz". For 4D variables, the name could be only "bv". The
!   meaning of "bv", "hs', "th", "tp" or "tz" are "Mixing coefficients induced
!   by non-breaking wave", "Significant wave height", "Mean wave direction",
!   "Zero-crossing wave period", and "Spectrum peak wave period" respectively.
!   This name is case-insensitive but must not contain blank space characters.
!\\\\
!EOP
    ierr=0;var=0
    select case (vname)
    case ('BV','Bv','bV','bv')
      call mwmsub_getvar_cuv(bv=var)
    case ('HS','Hs','hS','hs')
      call mwmsub_getvar_blk(1,hs=var)
    case ('TH','Th','tH','th')
      call mwmsub_getvar_blk(1,th=var)
    case ('TP','Tp','tP','tp')
      call mwmsub_getvar_blk(1,tp=var)
    case ('TZ','Tz','tZ','tz')
      call mwmsub_getvar_blk(1,tz=var)
!... added by BaoYing, 20190103
    case ('US','Us','uS','us')
      call mwmsub_getvar_blk(1,us=var)
    case ('VS','Vs','vS','vs')
      call mwmsub_getvar_blk(1,vs=var)
!... end add 	
    case default
      ierr=1
    end select
  end subroutine mwmsub_getvar_byname_3d
!BOP
! !INTERFACE:
  subroutine mwmsub_getvar_byname_4d(var,vname,ierr)
! !INPUT PARAMETERS:
    character(len=*),intent(in) :: vname
! !OUTPUT PARAMETERS:
    real(8),intent(out) :: var(:,:,:,:)
    integer,intent(out) :: ierr
! !DESCRIPTION:
!   Get variables according to its name. For 2D variables, the name could be
!   "hs', "th", "tp" or "tz". For 3D variables, the name could be "bv", "hs',
!   "th", "tp" or "tz". For 4D variables, the name could be only "bv". The
!   meaning of "bv", "hs', "th", "tp" or "tz" are "Mixing coefficients induced
!   by non-breaking wave", "Significant wave height", "Mean wave direction",
!   "Zero-crossing wave period", and "Spectrum peak wave period" respectively.
!   This name is case-insensitive but must not contain blank space characters.
!\\\\
!EOP
    ierr=0;var=0
    select case (vname)
    case ('BV','Bv','bV','bv')
      call mwmsub_getvar_blk(1,bv=var)
    case default
      ierr=1
    end select
  end subroutine mwmsub_getvar_byname_4d
!-------------------------------------------------------------------------------
!BOP
! !INTERFACE:
  subroutine mwmsub_getvar_cuv(bv,& ! Mixing coefficients induced by non-breaking wave.
                               hs,& ! Significant wave height (m)
                               th,& ! Mean wave direction (Deg)
                               tp,& ! Zero-crossing wave period (s)
!... modified by BaoYing, 20190103
!                               tz ) ! Spectrum peak wave period (s)
                               tz,& ! Spectrum peak wave period (s)
                               us,& ! Stokes velocity, zonal (m/s)
                               vs)  ! Stokes velocity, meridional (m/s)
!... end mod

! !OUTPUT PARAMETERS:
    real(8),intent(out),optional :: bv(:,:,:) !(nx,ny,kb)
    ! ---- The mixing coefficients (bv) for coupling of the wave-circulation coupling
    real(8),intent(out),optional :: hs(:,:)   !(nx,ny)
    ! --- Significant wave height (m)
    real(8),intent(out),optional :: th(:,:)   !(nx,ny)
    ! --- Mean wave direction (Deg)
    real(8),intent(out),optional :: tp(:,:)   !(nx,ny)
    ! --- Spectrum peak wave period (s)
    real(8),intent(out),optional :: tz(:,:)   !(nx,ny)
    ! --- Zero-crossing wave period (s)
!... added by BaoYing, 20190103
    real(8),intent(out),optional :: us(:,:)   !(nx,ny)
    ! --- stokes velocity, zonal (m/s)
    real(8),intent(out),optional :: vs(:,:)   !(nx,ny)
    ! --- stokes velocity, meridional (m/s)
!... end add 	
!
! !DESCRIPTION:
!   Get variables for the coupling of wave and circulation, or get some variables of
!   the wave model for output.
!\\\\
!EOP
    integer :: idx
    if(present(bv))then
      if(int(count_steps)>0)then
        ea=ea/count_steps
        call intact_mixture_bv(kb,np,h3,bv0)
        ea=0.d0;count_steps=0.d0
      endif
      call gridtrans_3d_inv(bv0,bv)
    endif
    call gridtrans_2d_inv(h1_3,hs)
    call gridtrans_2d_inv(tpf ,tp)
    call gridtrans_2d_inv(ape ,tz)
    call gridtrans_2d_inv(aet ,th)
!... added by BaoYing, 20190103
    call gridtrans_2d_inv(ustokes ,us)
    call gridtrans_2d_inv(vstokes ,vs)
!... end add 	
  end subroutine mwmsub_getvar_cuv
!-------------------------------------------------------------------------------
!BOP
! !INTERFACE:
  subroutine mwmsub_getvar_blk(nblock,& ! Block size.
                               bv,& ! Non-breaking wave induced Mixing coefficients.
                               hs,& ! Significant wave height (m)
                               th,& ! Mean wave direction (Deg)
                               tp,& ! Zero-crossing wave period (s)
!... modified by BaoYing, 20190103
!                               tz ) ! Spectrum peak wave period (s)
                               tz,& ! Spectrum peak wave period (s)
                               us,& ! Stokes velocity, zonal (m/s)
                               vs)  ! Stokes velocity, meridional (m/s)
!... end mod

! !INPUT PARAMETERS:
    integer,intent(in) :: nblock
    ! --- Block size.
! !OUTPUT PARAMETERS:
    real(8),intent(out),optional :: bv(:,:,:,:) !(nx,ny,kb,nblock)
    ! ---- The mixing coefficients induced by nonebreaking wave. (m2/s)
    real(8),intent(out),optional :: hs(:,:,:)   !(nx,ny,nblock)
    ! --- Significant wave height (m)
    real(8),intent(out),optional :: th(:,:,:)   !(nx,ny,nblock)
    ! --- Mean wave direction (Deg)
    real(8),intent(out),optional :: tp(:,:,:)   !(nx,ny,nblock)
    ! --- Spectrum peak wave period (s)
    real(8),intent(out),optional :: tz(:,:,:)   !(nx,ny,nblock)
    ! --- Zero-crossing wave period (s)
!... added by BaoYing, 20190103
    real(8),intent(out),optional :: us(:,:,:)   !(nx,ny)
    ! --- stokes velocity, zonal (m/s)
    real(8),intent(out),optional :: vs(:,:,:)   !(nx,ny)
    ! --- stokes velocity, meridional (m/s)
!... end add 	
!
! !DESCRIPTION:
!   Get variables for the coupling of wave and circulation, or get some variables of
!   the wave model for output.
!\\\\
!EOP
    integer :: idx
    if(present(bv))then
      if(int(count_steps)>0)then
        ea=ea/count_steps
        call intact_mixture_bv(kb,np,h3,bv0)
        ea=0.d0;count_steps=0.d0
      endif
      call gridtrans_3d_inv_blk(bv0,bv)
    endif
    do idx=1,np
      h1_3(idx)=cnb(idx)%angle
    enddo
    call gridtrans_2d_inv_blk(h1_3,hs)
    call gridtrans_2d_inv_blk(tpf ,tp)
    call gridtrans_2d_inv_blk(ape ,tz)
    call gridtrans_2d_inv_blk(aet ,th)
!... added by BaoYing, 20190103
    call gridtrans_2d_inv_blk(ustokes ,us)
    call gridtrans_2d_inv_blk(vstokes ,vs)
!... end add 	
  end subroutine mwmsub_getvar_blk
!-------------------------------------------------------------------------------
!BOP
! !IROUTINE:  mwmsub_getsize
! !INTERFACE:
  subroutine mwmsub_getsize(ksize,jsize)
! !OUTPUT PARAMETERS:
    integer,intent(out) :: ksize  ! - grid size in dimension of wavenumber.
    integer,intent(out) :: jsize  ! - grid size in diemnsion of wave direction.
! !DESCRIPTION: Get the model size in physical space with the dimensions of
!  wavenumber and wave direction.
!\\\\
!EOP
    ksize=kl  ! - grid size in dimension of wavenumber.
    jsize=jl  ! - grid size in diemnsion of wave direction.
  end subroutine mwmsub_getsize
!-------------------------------------------------------------------------------
!-------------------------------------------------------------------------------
  subroutine set_model_size_and_var(tlon,tlat,mask,depth,halosize,inrbonds)
    real(8),intent(in) :: tlon(:,:,:)
    real(8),intent(in) :: tlat(:,:,:)
    integer,intent(in) :: mask(:,:,:)
    real(8),intent(in) :: depth(:,:,:)
    integer,optional,intent(in) :: halosize
    integer,optional,intent(in) :: inrbonds(4)
    integer :: np,npc,i,j,idx,is,ie,js,je,npall,iblock,k,ihalo
    integer,allocatable :: ij2s(:,:,:),npstart(:),nsp_(:)
    real(spdp),allocatable :: mlon(:,:),mlat(:,:)
    !------------------------------------------
    !------------------------------------------
    nx=size(mask,1)
    ny=size(mask,2)
    nblock=size(mask,3)
    npall=nx*ny*nblock
    !------------------------------------------
    halo_size=1;if(present(halosize))halo_size=halosize
    is=halo_size+1;ie=nx-halo_size;js=halo_size+1;je=ny-halo_size
    if(present(inrbonds))then
      is=inrbonds(1);ie=inrbonds(2);js=inrbonds(3);je=inrbonds(4)
    endif
    if(.not.allocated(np_halo))allocate(np_halo(halo_size))
    !------------------------------------------
    allocate(ij2s(0:nx+1,0:ny+1,nblock))
    allocate(pnb(0:npall))
    allocate(alon(0:npall))
    allocate(alat(0:npall))
    allocate(npstart(nblock))
    !------------------------------------------
    ij2s=0;np=0;npc=0;idx=0
    do iblock=1,nblock
      npstart(iblock)=idx+1
      do j=js,je
        do i=is,ie
          if(ij2s(i,j,iblock)>0 )cycle
          if(mask(i,j,iblock)==0)cycle
          idx=idx+1
          pnb(idx)%i=i
          pnb(idx)%j=j
          pnb(idx)%k=iblock
          ij2s(i,j,iblock)=idx
        enddo
      enddo
    enddo
    !------------------------------------------
    npc=idx
    !------------------------------------------
    do ihalo=1,halo_size
      np_halo(ihalo)=idx
      do iblock=1,nblock
        do j=1,ny
          do i=1,nx
            if(ij2s(i,j,iblock)>0 )cycle
            if(mask(i,j,iblock)==0)cycle
            if(i<is-ihalo .or. i>ie+ihalo)cycle
            if(j<js-ihalo .or. j>je+ihalo)cycle
            idx=idx+1
            pnb(idx)%i=i
            pnb(idx)%j=j
            pnb(idx)%k=iblock
            ij2s(i,j,iblock)=idx
          enddo
        enddo
      enddo
    enddo
    !------------------------------------------
    np=idx
    !------------------------------------------
    do idx=1,np
      i=pnb(idx)%i
      j=pnb(idx)%j
      k=pnb(idx)%k
      pnb(idx)%ul=ij2s(i-1,j+1,k);pnb(idx)%u=ij2s(i,j+1,k);pnb(idx)%ur=ij2s(i+1,j+1,k)
      pnb(idx)% l=ij2s(i-1,j  ,k);                         pnb(idx)% r=ij2s(i+1,j  ,k)
      pnb(idx)%dl=ij2s(i-1,j-1,k);pnb(idx)%d=ij2s(i,j-1,k);pnb(idx)%dr=ij2s(i+1,j-1,k)
    enddo
    !------------------------------------------
    call wamvar_mod_init(np,npc,nx,ny,kl,jl) ! --- init variables
    !------------------------------------------
    !---------------------------------------------------------------------
    ! --- required variables:  mlon,mlat,d,nsp
    nsp=0;d=0
    allocate(nsp_init(0:np))
    do idx=1,np
      alon(idx)=tlon (pnb(idx)%i,pnb(idx)%j,pnb(idx)%k)
      alat(idx)=tlat (pnb(idx)%i,pnb(idx)%j,pnb(idx)%k)
      d   (idx)=depth(pnb(idx)%i,pnb(idx)%j,pnb(idx)%k)
      nsp (idx)=mask (pnb(idx)%i,pnb(idx)%j,pnb(idx)%k)
    enddo
    nsp_init=nsp;grdflag=1
    !---------------------------------------------------------------------
    allocate(nsp_(np),mlon(nx,ny),mlat(nx,ny))
    do iblock=1,nblock
      nsp_=0
      do idx=1,np
        if(pnb(idx)%k==iblock)nsp_(idx)=1
      enddo
      mlon=tlon(:,:,iblock)
      mlat=tlat(:,:,iblock)
      call setgeoinf(mlon,mlat,np_halo(halo_size),nsp_)
    enddo
    deallocate(nsp_,mlon,mlat)
    !---------------------------------------------------------------------
  end subroutine set_model_size_and_var
!-------------------------------------------------------------------------------
  subroutine gridtrans_4d_inv_blk(varin,varout)
    real(spdp),intent(in) :: varin(:,:,:)              !(kl,jl,0:np)
    real(8),intent(out),optional :: varout(:,:,:,:,:)  !(nx,ny,kl,jl,nblock)
    integer :: idx
    if(.not. present(varout))return
    varout=0
    do idx=1,np
      varout(pnb(idx)%i,pnb(idx)%j,:,:,pnb(idx)%k)=varin(:,:,idx+1)
    enddo
  end subroutine gridtrans_4d_inv_blk

  subroutine gridtrans_3d_inv_blk(varin,varout)
    real(spdp),intent(in) :: varin(:,:)             !(kb,0:np)
    real(8),intent(out),optional :: varout(:,:,:,:) !(nx,ny,kb,nblock)
    integer :: idx
    if(.not. present(varout))return
    varout=0
    do idx=1,np
      varout(pnb(idx)%i,pnb(idx)%j,:,pnb(idx)%k)=varin(:,idx+1)
    enddo
  end subroutine gridtrans_3d_inv_blk

  subroutine gridtrans_2d_inv_blk(varin,varout)
    real(spdp),intent(in) :: varin(:)             !(0:np)
    real(8),intent(out),optional :: varout(:,:,:) !(nx,ny,nblock)
    integer :: idx
    if(.not. present(varout))return
    varout=0
    do idx=1,np
      varout(pnb(idx)%i,pnb(idx)%j,pnb(idx)%k)=varin(idx+1)
    enddo
  end subroutine gridtrans_2d_inv_blk

  subroutine gridtrans_4d_blk(varin,varout)
    real(8),intent(in) :: varin(:,:,:,:,:) !(nx,ny,kl,jl,nblock)
    real(spdp),intent(out) :: varout(:,:,:)      !(kl,jl,0:np)
    integer :: k,idx
    varout=0
    do idx=1,np
      varout(:,:,idx+1)=varin(pnb(idx)%i,pnb(idx)%j,:,:,pnb(idx)%k)
    enddo
  end subroutine gridtrans_4d_blk

  subroutine gridtrans_2d_blk(varin,varout)
    real(8),intent(in) :: varin(:,:,:)  !(nx,ny,nblock)
    real(spdp),intent(out) :: varout(:) !(0:np)
    integer :: idx
    varout=0
    do idx=1,np
      varout(idx+1)=varin(pnb(idx)%i,pnb(idx)%j,pnb(idx)%k)
    enddo
  end subroutine gridtrans_2d_blk
!-------------------------------------------------------------------------------
  subroutine gridtrans_4d_inv(varin,varout)
    real(spdp),intent(in) :: varin(:,:,:)            !(kl,jl,0:np)
    real(8),intent(out),optional :: varout(:,:,:,:)  !(nx,ny,kl,jl)
    integer :: idx
    if(.not. present(varout))return
    varout=0
    do idx=1,np
      varout(pnb(idx)%i,pnb(idx)%j,:,:)=varin(:,:,idx+1)
    enddo
  end subroutine gridtrans_4d_inv

  subroutine gridtrans_3d_inv(varin,varout)
    real(spdp),intent(in) :: varin(:,:)
    real(8),intent(out),optional :: varout(:,:,:)
    integer :: idx
    if(.not. present(varout))return
    varout=0
    do idx=1,np
      varout(pnb(idx)%i,pnb(idx)%j,:)=varin(:,idx+1)
    enddo
  end subroutine gridtrans_3d_inv

  subroutine gridtrans_2d_inv(varin,varout)
    real(spdp),intent(in) :: varin(:)
    real(8),intent(out),optional :: varout(:,:)
    integer :: idx
    if(.not. present(varout))return
    varout=0
    do idx=1,np
      varout(pnb(idx)%i,pnb(idx)%j)=varin(idx+1)
    enddo
  end subroutine gridtrans_2d_inv

  subroutine gridtrans_4d(varin,varout)
    real(8),intent(in) :: varin(:,:,:,:) !(nx,ny,kl,jl)
    real(spdp),intent(out) :: varout(:,:,:)    !(kl,jl,0:np)
    integer :: k,idx
    varout=0
    do idx=1,np
      varout(:,:,idx+1)=varin(pnb(idx)%i,pnb(idx)%j,:,:)
    enddo
  end subroutine gridtrans_4d

  subroutine gridtrans_2d(varin,varout)
    real(8),intent(in) :: varin(:,:)
    real(spdp),intent(out) :: varout(:)
    integer :: idx
    varout=0
    do idx=1,np
      varout(idx+1)=varin(pnb(idx)%i,pnb(idx)%j)
    enddo
  end subroutine gridtrans_2d
!-------------------------------------------------------------------------------
  !subroutine gridtrans_3d(varin,varout)
  !  real(spdp),intent(in) :: varin(:,:,:) !(im,jm,kb)
  !  real(8),intent(out) :: varout(:,:)    !(kb,0:np)
  !  integer :: k,idx
  !  varout=0
  !  do k=1,kb
  !    do idx=1,np
  !     varout(k,idx+1)=varin(pnb(idx)%i,pnb(idx)%j,k)
  !   enddo
  !  enddo
  !end subroutine gridtrans_3d
  !
  !subroutine gridtrans_3d_blk(varin,varout)
  !  real(spdp),intent(in) :: varin(:,:,:,:) !(nx,ny,kb,nblock)
  !  real(8),intent(out) :: varout(:,:)      !(kb,0:np)
  !  integer :: k,idx
  !  varout=0
  !  do idx=1,np
  !    varout(:,idx+1)=varin(pnb(idx)%i,pnb(idx)%j,:,pnb(idx)%k)
  !  enddo
  !end subroutine gridtrans_3d_blk
!-------------------------------------------------------------------------------
!BOP
! !IROUTINE: mwmsub_final
! !INTERFACE:
  subroutine mwmsub_final
!
! !DESCRIPTION:
!   This routine will release the memory occupied by this wave model.
!\\\\
!EOP
    call final_propagat
    call wamvar_mod_final
    !if(associated(ij2s))then;deallocate(ij2s);nullify(ij2s);endif
    if(allocated(nsp_init))deallocate(nsp_init)
    !test if(allocated(mlon))deallocate(mlon)
    !test if(allocated(mlat))deallocate(mlat)
    if(allocated(wx0))deallocate(wx0)
    if(allocated(wy0))deallocate(wy0)
    if(allocated(wx1))deallocate(wx1)
    if(allocated(wy1))deallocate(wy1)
    if(allocated(h3))deallocate(h3)
    if(allocated(bv0))deallocate(bv0)
    total_step  = 0
    isfirstwind = 1
    iscoolstart = 1
    ineedvecrot = 0
    wind_type   = 0
    count_steps = 0
  end subroutine mwmsub_final
!-------------------------------------------------------------------------------
!-------------------------------------------------------------------------------
  subroutine mwmsub_forward_noprgt(istep,       & ! Step index between 2 wind fields.
                            spectrum,    & ! Wave energy spectrum.
                            windx,windy, & ! Wind vector 10m above surface.
                            newmask,     & ! Ice mask for each grid, 0 or 1.
                            uvel,vvel    ) ! Circulation currents, m/s.
! !INPUT PARAMETERS:
    integer,intent(in) :: istep
    ! --- Index of steps within twice inquire of the mixing coefficients (bv)
    !     for coupling of the wave-circulation coupling. If 0 is given for it,
    !     there will be no wind interpolation along time and the wind provided
    !     into this subroutine will be used directly.
    real(8),intent(in) :: windx(:,:)              !(nx,ny)
    real(8),intent(in) :: windy(:,:)              !(nx,ny)
    ! --- Wind vectors on the time of inquiring the mixing coefficients (bv).
    !     The unit is m/s and they are at 10m above the sea surface with the
    !     same grid with the initialization.
    real(8),intent(in),optional :: newmask(:,:)   !(nx,ny)
    ! --- Mask fir ice coverage, 0 for land/ice grid and 1 for water grid.
    real(8),intent(in),optional :: uvel(:,:)      !(nx,ny)
    real(8),intent(in),optional :: vvel(:,:)      !(nx,ny)
    ! --- Background circulation (vertical averaged or at surface),
    !     the unit is m/s.
! !INPUT/OUTPUT PARAMETERS:
    real(8),intent(inout) :: spectrum(:,:,:,:)    !(nx,ny,kl,jl)
    ! --- Energy spectrum of surface wave. This will need to readin from restart file
    !     or output for restart the model.
!
! !DESCRIPTION: This routine will forward one step of the wave model. The wind vector
!  will be required, the ice mask and current velocity are optional. The vector could
!  be agree with north/east along the longitude/latitude, or along with the grid lines.
!  Windstreees are also available for the forcing of thie wave model.
!\\\\
!EOP
    !---------------------------------------------------------------------------
    real(8) :: windweight
    integer :: idx,ihalo
    !---------------------------------------------------------------------------
    call gridtrans_4d(spectrum,ee)        ! --- Transfer spectrum to model grid.
    if(iscoolstart==1)then                ! New add by yinxq,20190121
      call set_wind_data;return           ! --- Set wind data
    endif                                 ! End of New add by yinxq,20190121
    call set_wind_data                    ! --- Set wind data
    call set_new_mask(newmask)            ! --- Set ice mask
    !---------------------------------------------------------------------------
    do ihalo=halo_size,1,-1

      windweight=dble(istep)/dble(total_step)
      if(istep/=0)windweight=1.d0
      wx=windweight*wx1+(1.d0-windweight)*wx0
      wy=windweight*wy1+(1.d0-windweight)*wy0

      e=ee
      !do idx=1,np_halo(ihalo) !npc
      ! call propagat(idx)                ! --- For wave propagation.
      !enddo

      do idx=1,np_halo(ihalo) !npc
        call implsch(idx)                 ! --- Source functions.
      enddo
      !call smoth_ee                      ! --- Smooth the wave spectrum.
#ifdef USEDIRSMOTH
      call smoth_ee_dir
#endif
    enddo
    !---------------------------------------------------------------------------
    call gridtrans_4d_inv(ee,spectrum)    ! --- Transfer spectrum to model grid.
    !---------------------------------------------------------------------------
    !!ea=ea+ee;count_steps=count_steps+1.d0 ! --- Forward wave model
    !---------------------------------------------------------------------------
    contains
    !---------------------------------------------------------------------------
    subroutine set_wind_data
      integer :: idx
      !----------------------------------------
      if(istep==total_step .or. istep==0)then
        wx0=wx1;wy0=wy1
      endif
      call gridtrans_2d(windx,wx)
      call gridtrans_2d(windy,wy)
      if(ineedvecrot==1)then
        do idx=1,np
          wx1(idx)=wx(idx)*cos(cnb(idx)%angle)-wy(idx)*sin(cnb(idx)%angle)
          wy1(idx)=wy(idx)*cos(cnb(idx)%angle)+wx(idx)*sin(cnb(idx)%angle)
        enddo
      else
        wx1=wx;wy1=wy
      endif
      if(isfirstwind==1)then
        wx0=wx1;wy0=wy1;isfirstwind=0
      endif
      !----------------------------------------
      ! --- Set for cool start
      if(iscoolstart==1)then
        wx=wx0;wy=wy0
        do idx=1,np
          call setspec(1,idx)
        enddo
        iscoolstart=0
        write(*,*)'Initial EE',minval(ee),maxval(ee)
      endif
      !----------------------------------------
      !windweight=dble(istep)/dble(total_step)
      !if(istep/=0)windweight=1.d0
      !wx=windweight*wx1+(1.d0-windweight)*wx0
      !wy=windweight*wy1+(1.d0-windweight)*wy0
    end subroutine set_wind_data
    !---------------------------------------------------------------------------
    subroutine set_new_mask(newmask)
      real(8),intent(in),optional :: newmask(:,:)   !(nx,ny)
      if(present(newmask))then
        do idx=1,np
          nsp(idx)=nsp_init(idx)*newmask(pnb(idx)%i,pnb(idx)%j)
        enddo
      endif
    end subroutine set_new_mask
  end subroutine mwmsub_forward_noprgt
!-------------------------------------------------------------------------------
  end module mwmsub_mod
!-------------------------------------------------------------------------------
