!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

 module mwave

! MODULE: mwave

! DESCRIPTION:
! This is the main driver module for masnum wave model.
! It contains necessary forcing fields and output fields
! as well as ncecessary routines for call proper initialization
! and update routines for masnum.

! HISTORY
! Created by Zhenya.Song, 2018/05/07


! USES
    use POP_ErrorMod
    use POP_FieldMod
    use POP_GridHorzMod
    use POP_KindsMod
    use kinds_mod, only: log_kind, int_kind, char_len, r8
    use constants, only: c0, c1, delim_fmt, blank_fmt, field_loc_center, &
                         field_type_scalar,radian,mpercm,cmperm,         &
                         char_blank
    use blocks, only: nx_block, ny_block, nghost
!    use distribution
    use domain, only: nblocks_clinic, POP_haloClinic
    use domain_size, only: km, nx_global, ny_global
    use grid, only: zt, KMT, TLON, TLAT, HT, ugrid_to_tgrid
    use grid, only: zw ! changed by shuqi in 20181224
    use POP_HaloMod, only: POP_HaloUpdate
!    use boundary, only: update_ghost_cells
!    use io_types, only: nml_in, nml_filename, stdout
    use forcing_fields, only: IFRAC, U10_SQR, SMF, SMFT, lsmft_avail
    use tavg, only: define_tavg_field, accumulate_tavg_field,          &
                    accumulate_tavg_now 
    use time_management, only: steps_per_day, dtt
    use exit_mod, only: sigAbort, exit_POP
    use broadcast, only: broadcast_scalar
    use communicate, only: my_task, master_task, MPI_COMM_OCN
    use io
    use io_types
    use mwmsub_mod
        
    
    implicit none
    private
    save
    
! PUBLIC MEMBER FUNCTIONS:

    public :: mwave_init1,         &
              mwave_init2,         &
              mwave_driver,        &
              mwave_getvar,        &
              mwave_read_restart, &
              mwave_write_restart
    
    
    real (r8), dimension(:,:,:,:), allocatable, public, target :: &
      TBV_w     ! wave-induced mixing

    real (r8), dimension(:,:,:), allocatable, public, target ::   &
      HS_w      ! significant wave height

!... added by BaoYing, 20190103
    real (r8), dimension(:,:,:), allocatable, public, target ::   &
      US_w,&    ! stokes velocity, zonal
      VS_w      ! stokes velocity, meridional
!... end add 	


    real (r8), public ::          &
      temp_coef,                  &! coefficient for temperature of bv
      salt_coef,                  &! coefficient for salinity of bv
      mome_coef                    ! coefficient for momentum of bv

    logical (log_kind), public :: &
      mwave_run           ! logical flags for masnum wave run

!... added by BaoYing, 20190103
    logical (log_kind), public :: &
      l_cpl_stokes,&      ! logical flags for effects of stokes drifts on air-sea fluxes
      l_cpl_spray         ! logical flags for effects of spray on air-sea fluxes
!... end add 	
    
!--------------------------------------------------------------------
!
!  local variables
!
!--------------------------------------------------------------------
    integer (int_kind) :: &
      start_w             ! start type, 0/1 is startup(default)/continute
    
    logical (log_kind) :: &
      first_run_w         ! logical flags for masnum wave run

    integer (int_kind)  :: &
      num_w,               &! number of wave number
      dir_w                 ! number of wave direction

    integer (int_kind) :: &
      nstep_w              ! number of steps for wave model in a loop
    
    integer (int_kind) :: &
      k_bv_w               ! max depth level for bv

    integer (int_kind) :: &
      wnd_type,           &! 0: 10m wind (default), 1: windstress 
      vec_dir              ! 0: vector direction same with ocean (default)
                           ! 1: lon/lat wind direction    

    integer (int_kind) :: &
      tavg_HS,            &! tavg_id for significant wave height
!... added by BaoYing, 20190103
      tavg_US,            &! tavg_id for stokes velocity, zonal
      tavg_VS,            &! tavg_id for stokes velocity, meridional
!... end add 	
      tavg_TH,            &! tavg_id for mean wave direction
      tavg_TP,            &! tavg_id for spectrum peak wave period
      tavg_TZ,            &! tavg_id for zero-crossing wave period
      tavg_TBV,           &! tavg_id for wave-induced mixing (bv)
      tavg_windx,         &! tavg_id for 10m wind
      tavg_windy           ! tavg_id for 10m wind

    integer (int_kind) :: &
      n_couple_w           ! Numerber of ocean steps for
                           ! wave and ocean coupling
    integer (int_kind) :: &
      step_w               ! total number of wave_driver run 

    integer (int_kind) :: &
      tmp_num_w,          &! temporary wave number, wave direction
      tmp_dir_w            ! get from restart

    real (r8) ::          &
      dep_bv_w             ! max depth for bv (m)

    real (r8) ::          &
      t_couple_w           ! Time (Hours) of wave and ocean coupling
      
    real (r8) ::          &
      nbound_w,           &! North bound for wave model (deg)
      sbound_w             ! South bound for wave model (deg)
      
    real (r8) ::          &
      ifrac_th_w           ! threshold for sea ice coverage
                           ! Lager than ifrac_w, set to land 
      
    integer (int_kind), dimension(:,:,:), allocatable, target :: &
      mask_w               ! Mask for wave, 0/1 means land/water 
                          
    real (r8), dimension(:,:,:), allocatable, target :: &
      wind_x0, wind_y0, wind_x1, wind_y1,        &! 10m wind fields
      taux, tauy                                  ! wind stress

    real (r8), dimension(:,:,:), allocatable, target :: &
!      HS_w,        &! significant wave height
      TH_w,        &! mean wave direction
      TP_w,        &! spectrum peak wave period
      TZ_w          ! zero-crossing wave period

    real (r8), dimension(:,:,:,:,:), allocatable, target ::      &
      spec_w,     &  ! wave spectrum                          &
      spec_w_tmp  ! wave spectrum for smooth

    integer(POP_i4) :: errorCode
!    real (r8), dimension(:,:,:,:,:), allocatable ::      &
!      tmp_ee    ! wave-induced mixing     

!********************************************************************

 contains
 
!********************************************************************
! ROUTINE: mwave_init1
! INTERFACE:

    subroutine mwave_init1

! DESCRIPTION:
! Initialize various quantities of masnum wave

! HISTORY:
! same as module
   include 'mpif.h'
!--------------------------------------------------------------------
!
!  local variables
!
!--------------------------------------------------------------------

    integer (int_kind) :: &
      i, j, k,            &! tmp loop index
      iblock               ! block index

    integer (int_kind) :: &
      nu,                 &! i/o unit
      nml_error,          &! namelist i/o error flag
      ierr                 ! MPI error flag
      
    character (char_len) :: outstring
     
!... modified by BaoYing, 20190103
!    add flag for stokes drift and spray which will affect the air-sea fluxes
    namelist /mwave_nml/ mwave_run,start_w,num_w,dir_w,nstep_w,   &
                         dep_bv_w, nbound_w,sbound_w, ifrac_th_w, &
                         n_couple_w,temp_coef,salt_coef,mome_coef, &
						 l_cpl_stokes, l_cpl_spray 
!... end mod 
!--------------------------------------------------------------------
!
!  set step_w to zero when initialaztin
!
!--------------------------------------------------------------------

    step_w = 0
    first_run_w = .true.

!--------------------------------------------------------------------
!
!  set wind type and direction
!  wind type (wnd_type) set to 0, use 10m wind
!  wind direction (vec_dir) set to 0, same with vector of ocean
!
!--------------------------------------------------------------------
  
    wnd_type = 0
    vec_dir  = 1 
    
!--------------------------------------------------------------------
!
!  set default vale, read input namelist and set wave run type
!
!--------------------------------------------------------------------
    
    mwave_run = .true.
    start_w   = 0      ! startup run (default)
    num_w     = 25     ! wave number: 25 (default)
    dir_w     = 24     ! wave direction: 24 (default)
    dep_bv_w  = 1000.  ! Maximum depth for bv cal: 1000m (default)
    nbound_w  = 88.    ! North boundary for wave model: 88deg (default)
    sbound_w  = -88.   ! South boundary for wave model: -88deg (default)
    ifrac_th_w = 0.3   ! Threshold for ice coverage: 0.3 (default)
    n_couple_w = 1     ! Frequency of wave ocean coupling: 1 (default)
    temp_coef = c1     ! the factor of bv for temperature: 1.0 (default)
    salt_coef = c1     ! the factor of bv for salinity: 1.0 (default)  
    mome_coef = c1     ! the factor of bv for momentum: 1.0 (default)
!... added by BaoYing, 20190103
    l_cpl_stokes = .true.  ! logical flags for stokes drift
    l_cpl_spray  = .true.  ! logical flags for spary
!... end add 	
    

    if (my_task == master_task) then
      open (nml_in, file=nml_filename, status='old',iostat=nml_error)
      if (nml_error /= 0) then
        nml_error = -1
      else
        nml_error = 1
      endif
      do while (nml_error > 0)
        read(nml_in, nml=mwave_nml, iostat=nml_error)
      enddo
      if (nml_error == 0) close(nml_in)  
    endif
    
    call broadcast_scalar(nml_error, master_task)
    if (nml_error /= 0) then
      call exit_POP(sigAbort, &
                    'ERROR reading mwave_nml')
    endif
    
    if (my_task == master_task) then
      write(stdout,delim_fmt)
      write(stdout,blank_fmt)
      write(stdout,'(a19)') 'MASNUM Wave options'
      write(stdout,blank_fmt)
      write(stdout,delim_fmt)
      
      if (mwave_run) then
        write(stdout, '(a15)') 'REAL MASNUM RUN'
      else
        write(stdout, '(a15)') 'STUB MASNUM RUN'
      endif

!--------------------------------------------------------------------
!
!  set the max depth level for bv cacluation
!
!--------------------------------------------------------------------      
      k_bv_w = km
      do k = 1, km-1
        !if (zt(k)*mpercm < dep_bv_w .and. & ! changed by shuqi in 20181224 
        !    zt(k+1)*mpercm > dep_bv_w) then ! changed by shuqi in 20181224
        !  k_bv_w = k                        ! changed by shuqi in 20181224
        !endif                               ! changed by shuqi in 20181224
        if (zw(k)*mpercm < dep_bv_w .and. &  ! changed by shuqi in 20181224
            zw(k+1)*mpercm > dep_bv_w) then  ! changed by shuqi in 20181224
          k_bv_w = k                         ! changed by shuqi in 20181224
        endif                                ! changed by shuqi in 20181224
      enddo
      
      write(stdout, '(a10,i2)')     'start_w = ', start_w
      write(stdout, '(a8,i2)')      'num_w = ', num_w
      write(stdout, '(a8,i2)')      'dir_w = ', dir_w
      write(stdout, '(a10,i2)')       'nstep_w = ', nstep_w
      write(stdout, '(a11,1pe10.3)')  'dep_bv_w = ', dep_bv_w
      write(stdout, '(a9,i2)')      'k_bv_w = ', k_bv_w
      write(stdout, '(a14,1pe10.3)')  'North bound = ', nbound_w
      write(stdout, '(a14,1pe10.3)')  'South bound = ', sbound_w
      write(stdout, '(a13,1pe10.3)')  'ifrac_th_w = ', ifrac_th_w
      write(stdout, '(a13,1pe10.3)')  'n_couple_w = ', n_couple_w
      write(stdout, '(a12,1pe10.3)')  'temp_coef = ', temp_coef
      write(stdout, '(a12,1pe10.3)')  'salt_coef = ', salt_coef
      write(stdout, '(a12,1pe10.3)')  'mome_coef = ', mome_coef
      write(stdout, '(a18, 1pe12.6)') 'Ocean step time = ', dtt
    endif

!--------------------------------------------------------------------
!
!  broadcast the variables
!
!--------------------------------------------------------------------
    
    call broadcast_scalar(mwave_run, master_task)
    call broadcast_scalar(start_w,   master_task)
    call broadcast_scalar(num_w,     master_task)
    call broadcast_scalar(dir_w,     master_task)
    call broadcast_scalar(nstep_w,   master_task)
    call broadcast_scalar(dep_bv_w,  master_task)
    call broadcast_scalar(k_bv_w,    master_task)
    call broadcast_scalar(nbound_w,  master_task)
    call broadcast_scalar(sbound_w,  master_task)
    call broadcast_scalar(temp_coef, master_task)
    call broadcast_scalar(salt_coef, master_task)
    call broadcast_scalar(mome_coef, master_task)
    call broadcast_scalar(ifrac_th_w,master_task)
    call broadcast_scalar(n_couple_w,master_task)
    call broadcast_scalar(wnd_type,  master_task)
    call broadcast_scalar(vec_dir,   master_task)
!... added by BaoYing, 20190103
    call broadcast_scalar(l_cpl_stokes,  master_task)
    call broadcast_scalar(l_cpl_spray,   master_task)
!... end add  

!--------------------------------------------------------------------
!
!  allocate variables and then set to zeros
!
!--------------------------------------------------------------------
    allocate (HS_w(nx_block,ny_block,nblocks_clinic))
!... added by BaoYing, 20190103
    allocate (US_w(nx_block,ny_block,nblocks_clinic))
    allocate (VS_w(nx_block,ny_block,nblocks_clinic))
!... end add 	
    allocate (TH_w(nx_block,ny_block,nblocks_clinic))
    allocate (TP_w(nx_block,ny_block,nblocks_clinic))
    allocate (TZ_w(nx_block,ny_block,nblocks_clinic))
    
    allocate (TBV_w(nx_block,ny_block,km,nblocks_clinic))
    
    allocate (spec_w(nx_block,ny_block,num_w,dir_w,nblocks_clinic))
    allocate (spec_w_tmp(nx_block,ny_block,num_w,dir_w,nblocks_clinic))
    
    allocate (wind_x0(nx_block,ny_block,nblocks_clinic))
    allocate (wind_y0(nx_block,ny_block,nblocks_clinic))
    allocate (wind_x1(nx_block,ny_block,nblocks_clinic))
    allocate (wind_y1(nx_block,ny_block,nblocks_clinic))
    
    allocate (mask_w(nx_block,ny_block,nblocks_clinic))
    
    allocate(taux(nx_block,ny_block,nblocks_clinic))
    allocate(tauy(nx_block,ny_block,nblocks_clinic))
    
    HS_w(:,:,:)       = c0
!... added by BaoYing, 20190103
    US_w(:,:,:)       = c0
    VS_w(:,:,:)       = c0
!... end add 	
    TH_w(:,:,:)       = c0
    TP_w(:,:,:)       = c0
    TZ_w(:,:,:)       = c0
    TBV_w(:,:,:,:)    = c0
    spec_w(:,:,:,:,:) = c0
    spec_w_tmp(:,:,:,:,:) = c0
    wind_x0(:,:,:)    = c0
    wind_y0(:,:,:)    = c0
    wind_x1(:,:,:)    = c0
    wind_y1(:,:,:)    = c0
    taux(:,:,:)       =c0
    tauy(:,:,:)       =c0 
    mask_w(:,:,:)     = 0
    
!--------------------------------------------------------------------
!
!  define fields for accumulating tavg dianostics
!
!--------------------------------------------------------------------

    call define_tavg_field(tavg_TBV,'TBV_w',3,                    &
                           long_name='Wave-induced mixing',       &
                           units='cm^2/s', grid_loc='3113')

    call define_tavg_field(tavg_HS,'HS_w',2,                      &
                           long_name='Significant wave height',   &
                           units='m', grid_loc='2110')

!... added by BaoYing, 20190103
    call define_tavg_field(tavg_US,'VS_w',2,                      &
                           long_name='stokes velocity, zonal',   &
                           units='m/s', grid_loc='2110')

    call define_tavg_field(tavg_VS,'US_w',2,                      &
                           long_name='stokes velocity, meridional',   &
                           units='m/s', grid_loc='2110')
!... end add 	

    call define_tavg_field(tavg_TH,'TH_w',2,                      &
                           long_name='Mean wave direction',       &
                           units='deg', grid_loc='2110')

    call define_tavg_field(tavg_TP,'TP_w',2,                      &
                           long_name='Spectrum peak wave period', &
                           units='s', grid_loc='2110')

    call define_tavg_field(tavg_TZ,'TZ_w',2,                      &
                           long_name='zero-crossing wave period', &
                           units='s', grid_loc='2110')

    call define_tavg_field(tavg_windx,'WINDX_w',2,                &
                           long_name='10m windx for wave model',  &
                           units='m/s', grid_loc='2110')
                           
    call define_tavg_field(tavg_windy,'WINDY_w',2,                &
                           long_name='10m windy for wave model',  &
                           units='m/s', grid_loc='2110')                           

 
!--------------------------------------------------------------------



!--------------------------------------------------------------------

 end subroutine mwave_init1 


!********************************************************************
! ROUTINE: mwave_init2
! INTERFACE:

    subroutine mwave_init2

! DESCRIPTION:
! Initialize restart, and 
! necessary forcing by call a separate routines for wind, mask

! HISTORY:
! same as module
   include 'mpif.h'
!--------------------------------------------------------------------
!
!  local variables
!
!--------------------------------------------------------------------

    integer (int_kind) :: &
      i, j, k,            &! tmp loop index
      iblock               ! block index
      
    integer (int_kind) :: &
      tmp_nstep_w,        &! temporary number of step, get new value 
      tmp_nstep_w1         ! from wave init, and then compare with 
                           ! nstep_w

    integer (int_kind) :: &
      nu,                 &! i/o unit
      nml_error,          &! namelist i/o error flag
      ierr                 ! MPI error flag
    
    real (r8) ::          &
      spd_tmp              !temp var for wind speed 
      
    character (char_len) :: outstring
 
!--------------------------------------------------------------------
!
!  initialize the masnum initialization routine
!
!--------------------------------------------------------------------
     
     !*** Input parameter for wave routine initialization
     ! TLON, TLAT: {longitude, latitude} of T points (radian), 
     !             should convert to degress
     ! HT: ocean depth at T points (cm), should convert to m
     ! zt: vert dist from sfc to midpoint of layer (cm), should 
     !     convert to m
     ! dtt: step time (s)
     ! nstep_w and tmp_nstep_w: number of wave integrate 
     !                          in one ocean loop
     ! mask_w: mask for wave model, controlled by nbound_w, sbound_w,
     !         and IFRAC (sea ice coverage), 0/1 means land/water
     ! num_w: wave number (default: 25)
     ! dir_w: wave direction (default: 24)
     ! wnd_type: wind type (default: 0, use 10m wind)
     ! vec_dir: wind direction (default: 0, same with vector of ocean)
     ! start_w: wave model run type (default: 0, startup run)



     !*** Get the mask from KMT for wave model, 0/1 means land/water
     ! Then remove the Isolated water point
     
     do iblock = 1, nblocks_clinic
       where (KMT(:,:,iblock) >= 1) mask_w(:,:,iblock) = 1
       where (TLAT(:,:,iblock)*radian > nbound_w .or. &
              TLAT(:,:,iblock)*radian < sbound_w)
              mask_w(:,:,iblock) = 0
       endwhere
      
!       where (IFRAC(:,:,iblock) >= ifrac_th_w) mask_w(:,:,iblock) = 0
      
       do j = 1 + nghost, ny_block-nghost
       	 do i = 1 + nghost, nx_block-nghost
       		 if (mask_w(i,j,iblock) == 1 .and. mask_w(i-1,j,iblock) == 0  &
       		    .and. mask_w(i+1,j,iblock) == 0                         &
       		    .and. mask_w(i,j-1,iblock) == 0                         &
       		    .and. mask_w(i,j+1,iblock) == 0 ) then
       		   mask_w(i,j,iblock) = 0
       		 endif
       	 enddo
       enddo
     enddo


    !*** Wave Ocean coupling frequency check

    if (mod(steps_per_day, n_couple_w * 1.0) == 0) then
   	  t_couple_w = n_couple_w * dtt / 3600
    else
   	  write(outstring,*) 'Frequency of wave ocean coupling ('         &
                         ,n_couple_w,                                 &
   	                     'Hours, does not fit for ocean step time (', &
                          steps_per_day,')'
      call exit_POP(sigAbort, trim(outstring))
    endif

     !*** get the 10m wind
     if (lsmft_avail) then
     	 taux(:,:,:) = SMFT(:,:,1,:)
     	 tauy(:,:,:) = SMFT(:,:,2,:)
     else
     	 do iblock = 1, nblocks_clinic
     	 	 call ugrid_to_tgrid(taux(:,:,iblock),SMF(:,:,1,iblock),iblock)
     	 	 call ugrid_to_tgrid(tauy(:,:,iblock),SMF(:,:,2,iblock),iblock)
     	 enddo
         call POP_HaloUpdate(taux, POP_haloClinic,           &
                             POP_gridHorzLocCenter,          &
                             POP_fieldKindVector, errorCode, &
                             fillValue = 0.0_POP_r8)
         if (errorCode /= POP_Success) then
           call POP_ErrorSet(errorCode, &
                  'mwave: error updating halo for taux')
           return
         endif
         call POP_HaloUpdate(tauy, POP_haloClinic,           &
                             POP_gridHorzLocCenter,          &
                             POP_fieldKindVector, errorCode, &
                             fillValue = 0.0_POP_r8)
         if (errorCode /= POP_Success) then
           call POP_ErrorSet(errorCode, &
                  'mwave: error updating halo for tauy')
           return
         endif
     endif
     
     do iblock = 1, nblocks_clinic
       do j = 1, ny_block
         do i = 1, nx_block
           spd_tmp = sqrt(taux(i,j,iblock)**2+tauy(i,j,iblock)**2)
           if (spd_tmp > c0) then
             wind_x0(i,j,iblock)  = taux(i,j,iblock) * sqrt(U10_SQR(i,j,iblock)) &
                                  / spd_tmp * mpercm
             wind_y0(i,j,iblock)  = tauy(i,j,iblock) * sqrt(U10_SQR(i,j,iblock)) &
                                  / spd_tmp * mpercm
           else
             wind_x0(i,j,iblock) = c0
             wind_y0(i,j,iblock) = c0
           endif
         enddo
       enddo
     enddo

     !*** Call the MASNUM Wave initialization routine
    
     tmp_nstep_w = nstep_w

       call mwmsub_init(tlon=TLON(:,:,:)*radian,              &
                        tlat=TLAT(:,:,:)*radian,              &
                        mask=mask_w(:,:,:),                    &
                        depth=HT(:,:,:)*mpercm,               &
                        !zlevel=zt(1:k_bv_w)*mpercm,               & !changed by shuqi in 20181224 
                        zlevel=zw(1:k_bv_w)*mpercm,                & !changed by shuqi in 20181224
                        nstep=tmp_nstep_w,                         &
                        delttime=dtt,ksize=num_w,jsize=dir_w,      &
                        vecdir=vec_dir,istart=start_w,             &
                        halosize=nghost)                           
!                        halosize=nghost,                           &
!                        windx=wind_x0(:,:,:),                 &
!                        windy=wind_y0(:,:,:))
!                        vec_dir, wnd_type, start_w)
     if (my_task == master_task) then
       write(stdout,*) 'nstep_w from wave model is ', tmp_nstep_w
     endif

!--------------------------------------------------------------------
!
!  MASNUM wave model step time check
!
!--------------------------------------------------------------------
    call MPI_ALLREDUCE(tmp_nstep_w, tmp_nstep_w1, 1, MPI_INTEGER, &
                        MPI_MAX, MPI_COMM_OCN, ierr)
    if (tmp_nstep_w1 > nstep_w) then
      write(outstring,*) 'nstep_w is too small: increase to ', &
                          tmp_nstep_w1
      call exit_POP(sigAbort, trim(outstring))
    endif
     
     
!--------------------------------------------------------------------



!--------------------------------------------------------------------

 end subroutine mwave_init2

!******************************************************************** 
! ROUTINE: mwave_driver
! INTERFACE:

    subroutine mwave_driver

!--------------------------------------------------------------------
!
!  local variables
!
!--------------------------------------------------------------------
    integer (int_kind) :: &
      n,n1,n2,n3,                  &! index of steps for wave model in one loop
      k, i, j,            &! tmp loop index
      iblock,             &!block index
      smooth_w   ! smooth specturm weight
    
    real (r8) ::    &
      spd_tmp        ! tmp for wind speed  
    
    step_w = step_w + 1
    
    if (mwave_run) then
!      if (my_task == master_task) then
!        write(stdout,'(a25)') 'MASNUM wave model running'
!      endif
      ! SMF is the surface momentum fluxes
      ! IFRAC is the ice fraction
      ! If Sea Ice coverage is more than 0.3, set to land
      ! Then remove the Isolated water point
!      if (start_w == 1 .and. first_run_w == .true.) then
      do iblock = 1, nblocks_clinic
        where (KMT(:,:,iblock) >= 1) mask_w(:,:,iblock) = 1
        where (TLAT(:,:,iblock)*radian > nbound_w .or. &
              TLAT(:,:,iblock)*radian < sbound_w)
              mask_w(:,:,iblock) = 0
        endwhere

        where (IFRAC(:,:,iblock) >= ifrac_th_w) mask_w(:,:,iblock) = 0

      	do j = 1 + nghost, ny_block-nghost
       	  do i = 1 + nghost, nx_block-nghost
       		if (mask_w(i,j,iblock) == 1 .and. mask_w(i-1,j,iblock) == 0  &
       		     .and. mask_w(i+1,j,iblock) == 0                         &
       		     .and. mask_w(i,j-1,iblock) == 0                         &
       		     .and. mask_w(i,j+1,iblock) == 0 ) then
       		    mask_w(i,j,iblock) = 0
       		  endif
       	  enddo
        enddo
      enddo
      
     !*** get the 10m wind
     if (lsmft_avail) then
     	 taux(:,:,:) = SMFT(:,:,1,:)
     	 tauy(:,:,:) = SMFT(:,:,2,:)
     else
     	 do iblock = 1, nblocks_clinic
     	 	 call ugrid_to_tgrid(taux(:,:,iblock),SMF(:,:,1,iblock),iblock)
     	 	 call ugrid_to_tgrid(tauy(:,:,iblock),SMF(:,:,2,iblock),iblock)
     	 enddo
         call POP_HaloUpdate(taux, POP_haloClinic,           &
                             POP_gridHorzLocCenter,          &
                             POP_fieldKindVector, errorCode, &
                             fillValue = 0.0_POP_r8)
         if (errorCode /= POP_Success) then
           call POP_ErrorSet(errorCode, &
                  'mwave: error updating halo for taux')
           return
         endif
         call POP_HaloUpdate(tauy, POP_haloClinic,           &
                             POP_gridHorzLocCenter,          &
                             POP_fieldKindVector, errorCode, &
                             fillValue = 0.0_POP_r8)
         if (errorCode /= POP_Success) then
           call POP_ErrorSet(errorCode, &
                  'mwave: error updating halo for tauy')
           return
         endif
     endif
     
     do iblock = 1, nblocks_clinic
       do j = 1, ny_block
         do i = 1, nx_block
           spd_tmp = sqrt(taux(i,j,iblock)**2+tauy(i,j,iblock)**2)
           if (spd_tmp > c0) then
             wind_x1(i,j,iblock)  = taux(i,j,iblock) * sqrt(U10_SQR(i,j,iblock)) &
                                  / spd_tmp * mpercm
             wind_y1(i,j,iblock)  = tauy(i,j,iblock) * sqrt(U10_SQR(i,j,iblock)) &
                                  / spd_tmp * mpercm
           else
             wind_x1(i,j,iblock) = c0
             wind_y1(i,j,iblock) = c0
           endif
         enddo
       enddo
     enddo
      
      do n = 1, nstep_w   
        ! call MASNUM wave model integrate routine
        call mwmsub_forward(istep=n, spectrum=spec_w(:,:,:,:,:), &
                            windx=wind_x1(:,:,:),                &
                            windy=wind_y1(:,:,:),                &
                            newmask=mask_w(:,:,:)*c1)
        ! update the wind at old time
        wind_x0 = wind_x1
        wind_y0 = wind_y1
        
        ! update the boundary
!        call update_ghost_cells(spec_w(:,:,:,:,:), bndy_clinic, &
!                           field_loc_center, field_type_scalar)
        call POP_HaloUpdate(spec_w(:,:,:,:,:),POP_haloClinic, &
                            POP_gridHorzLocCenter,            &
                            POP_fieldKindScalar, errorCode,   &
                            fillValue = 0.0_POP_r8)
        if (errorCode /= POP_Success) then
        	call POP_ErrorSet(errorCode, &
        	   'mwave: error updating halo for spectrum')
        	return
        endif
!     smooth the wave spectrum   
        spec_w_tmp = spec_w   
        do iblock = 1, nblocks_clinic
        	do j = 1 + nghost, ny_block-nghost
         	  do i = 1 + nghost, nx_block-nghost
         		if (mask_w(i,j,iblock) == 1 ) then
       		    spec_w_tmp(i,j,:,:,iblock) = spec_w(i,j,:,:,iblock) * 48 
       		    smooth_w = 48
       		    if (mask_w(i-1,j-1,iblock) == 1) then
       		    	spec_w_tmp(i,j,:,:,iblock) =     &
       		    	 spec_w_tmp(i,j,:,:,iblock) + spec_w(i-1,j-1,:,:,iblock)
       		    	 smooth_w = smooth_w + 1
       		    endif
       		    if (mask_w(i,j-1,iblock) == 1) then
       		    	spec_w_tmp(i,j,:,:,iblock) =     &
       		    	 spec_w_tmp(i,j,:,:,iblock) + spec_w(i,j-1,:,:,iblock)
       		    	 smooth_w = smooth_w + 1
       		    endif	
       		    if (mask_w(i+1,j-1,iblock) == 1) then
       		    	spec_w_tmp(i,j,:,:,iblock) =     &
       		    	 spec_w_tmp(i,j,:,:,iblock) + spec_w(i+1,j-1,:,:,iblock)
       		    	 smooth_w = smooth_w + 1
       		    endif
       		    if (mask_w(i+1,j,iblock) == 1) then
       		    	spec_w_tmp(i,j,:,:,iblock) =     &
       		    	 spec_w_tmp(i,j,:,:,iblock) + spec_w(i+1,j,:,:,iblock)
       		    	 smooth_w = smooth_w + 1
       		    endif
       		    if (mask_w(i+1,j+1,iblock) == 1) then
       		    	spec_w_tmp(i,j,:,:,iblock) =     &
       		    	 spec_w_tmp(i,j,:,:,iblock) + spec_w(i+1,j+1,:,:,iblock)
       		    	 smooth_w = smooth_w + 1
       		    endif
       		    if (mask_w(i,j+1,iblock) == 1) then
       		    	spec_w_tmp(i,j,:,:,iblock) =     &
       		    	 spec_w_tmp(i,j,:,:,iblock) + spec_w(i,j+1,:,:,iblock)
       		    	 smooth_w = smooth_w + 1
       		    endif
       		    if (mask_w(i-1,j+1,iblock) == 1) then
       		    	spec_w_tmp(i,j,:,:,iblock) =     &
       		    	 spec_w_tmp(i,j,:,:,iblock) + spec_w(i-1,j+1,:,:,iblock)
       		    	 smooth_w = smooth_w + 1
       		    endif
       		    if (mask_w(i-1,j,iblock) == 1) then
       		    	spec_w_tmp(i,j,:,:,iblock) =     &
       		    	 spec_w_tmp(i,j,:,:,iblock) + spec_w(i-1,j,:,:,iblock)
       		    	 smooth_w = smooth_w + 1
       		    endif
       		  endif
       		  spec_w_tmp(i,j,:,:,iblock) = spec_w_tmp(i,j,:,:,iblock)/smooth_w
         	  enddo
          enddo
        enddo      	
        spec_w = spec_w_tmp
        call POP_HaloUpdate(spec_w(:,:,:,:,:),POP_haloClinic, &
                            POP_gridHorzLocCenter,            &
                            POP_fieldKindScalar, errorCode,   &
                            fillValue = 0.0_POP_r8)
        if (errorCode /= POP_Success) then
        	call POP_ErrorSet(errorCode, &
        	   'mwave: error updating halo for spectrum')
        	return
        endif
      enddo
          
      call mwave_getvar      

      first_run_w = .false.

    else
      ! Only stub wave model running
      if (my_task == master_task) then
        write(stdout,'(a25)') 'Stub wave model running'
      endif
      ! set TBV_w, HS_w, TH_w, TP_w, TZ_w to zero
      TBV_w = c0
      HS_w  = c0
!... added by BaoYing, 20190103
      US_w  = c0
      VS_w  = c0
!... end add 	
      TH_w  = c0
      TP_w  = c0
      TZ_w  = c0
      
    endif


!--------------------------------------------------------------------

 end subroutine mwave_driver 

!******************************************************************** 
! ROUTINE: mwave_getvar
! INTERFACE:

    subroutine mwave_getvar

! DESCRIPTION:
! Drive the masnum wave with 10m wind, ice mask 
! If mwave_run is true, MASNUM wave model will integrate
! else, stub wave model will run, all variables related to wave 
! will be set to zero

! HISTORY:
! same as module
   include 'mpif.h'
!--------------------------------------------------------------------
!
!  local variables
!
!--------------------------------------------------------------------
    integer (int_kind) :: &
      i, j, k,            &! depth loop index
      iblock,             &! block index
      ierr,               &
      ierr_io,ierr1_io

    real (r8), dimension(:,:,:,:), allocatable :: &
      tmp_tbv  
  
    character (char_len) :: outstring

!--------------------------------------------------------------------
!
!  Get the wave model variables, TBV_w, controlled by n_couple_w
!
!--------------------------------------------------------------------
    !*** Get TBV_w,controlled by n_couple_w  
    if (mod(step_w, n_couple_w) == 0) then 
     
      allocate(tmp_tbv(nx_block,ny_block,k_bv_w,nblocks_clinic))

      ! Get the wave-induced mixing (bv)   
      tmp_tbv = c0
      ierr_io = 0
      ierr1_io = 0
      
      call mwmsub_getvar(var=tmp_tbv(:,:,:,:),vname='bv',ierr=ierr1_io)
   	  TBV_w(:,:,1:k_bv_w,:) = tmp_tbv(:,:,:,:) * cmperm *cmperm   	    
      deallocate(tmp_tbv)
      call MPI_ALLREDUCE(ierr1_io, ierr_io, 1, MPI_INTEGER, &
                         MPI_MAX, MPI_COMM_OCN, ierr)
      if (ierr_io == 1) then
        write(outstring,*) 'Geting bv is wrong, check the name'
        call exit_POP(sigAbort, trim(outstring))
      endif

!... added by BaoYing, 20190103
!--------------------------------------------------------------------
!
!  Get the wave model variables, US_w and VS_w
!
!--------------------------------------------------------------------

      ierr_io = 0
      ierr1_io = 0
      call mwmsub_getvar(var=US_w(:,:,:),vname='us',ierr=ierr1_io) 
      call MPI_ALLREDUCE(ierr1_io, ierr_io, 1, MPI_INTEGER, &
                         MPI_MAX, MPI_COMM_OCN, ierr)
      if (ierr_io == 1) then
        write(outstring,*) 'Geting us is wrong, check the name'
        call exit_POP(sigAbort, trim(outstring))
      endif

      ierr_io = 0
      ierr1_io = 0
      call mwmsub_getvar(var=VS_w(:,:,:),vname='vs',ierr=ierr1_io) 
      call MPI_ALLREDUCE(ierr1_io, ierr_io, 1, MPI_INTEGER, &
                         MPI_MAX, MPI_COMM_OCN, ierr)
      if (ierr_io == 1) then
        write(outstring,*) 'Geting vs is wrong, check the name'
        call exit_POP(sigAbort, trim(outstring))
      endif


!... end add 

    endif

	

!--------------------------------------------------------------------
!
!  Get the wave model variables, hs_w, th_w, tp_w, tz_w
!
!--------------------------------------------------------------------
    
    ierr_io = 0
    ierr1_io = 0
    call mwmsub_getvar(var=HS_w(:,:,:),vname='hs',ierr=ierr1_io) 
    call MPI_ALLREDUCE(ierr1_io, ierr_io, 1, MPI_INTEGER, &
                       MPI_MAX, MPI_COMM_OCN, ierr)
    if (ierr_io == 1) then
      write(outstring,*) 'Geting hs is wrong, check the name'
      call exit_POP(sigAbort, trim(outstring))
    endif


    ierr_io = 0
    ierr1_io = 0
    call mwmsub_getvar(var=TH_w(:,:,:),vname='th',ierr=ierr1_io)    	    	                    
    call MPI_ALLREDUCE(ierr1_io, ierr_io, 1, MPI_INTEGER, &
                       MPI_MAX, MPI_COMM_OCN, ierr)
    if (ierr_io == 1) then
      write(outstring,*) 'Geting th is wrong, check the name'
      call exit_POP(sigAbort, trim(outstring))
    endif


    ierr_io = 0
    ierr1_io = 0
    call mwmsub_getvar(var=TP_w(:,:,:),vname='tp',ierr=ierr1_io) 
    call MPI_ALLREDUCE(ierr1_io, ierr_io, 1, MPI_INTEGER, &
                       MPI_MAX, MPI_COMM_OCN, ierr)
    if (ierr_io == 1) then
      write(outstring,*) 'Geting tp is wrong, check the name'
      call exit_POP(sigAbort, trim(outstring))
    endif

    ierr_io = 0
    ierr1_io = 0
    call mwmsub_getvar(var=TZ_w(:,:,:),vname='tz',ierr=ierr1_io) 
    call MPI_ALLREDUCE(ierr1_io, ierr_io, 1, MPI_INTEGER, &
                       MPI_MAX, MPI_COMM_OCN, ierr)
    if (ierr_io == 1) then
      write(outstring,*) 'Geting tz is wrong, check the name'
      call exit_POP(sigAbort, trim(outstring))
    endif

!--------------------------------------------------------------------
!
!  Tavg output
!
!--------------------------------------------------------------------
    
    do iblock = 1, nblocks_clinic
      if (accumulate_tavg_now(tavg_HS)) then
        call accumulate_tavg_field(HS_w(:,:,iblock),tavg_HS,iblock,1)
      endif
!... added by BaoYing, 20190103
      if (accumulate_tavg_now(tavg_US)) then
        call accumulate_tavg_field(US_w(:,:,iblock),tavg_US,iblock,1)
      endif
      if (accumulate_tavg_now(tavg_VS)) then
        call accumulate_tavg_field(VS_w(:,:,iblock),tavg_VS,iblock,1)
      endif
!... end add 	
      if (accumulate_tavg_now(tavg_TH)) then
        call accumulate_tavg_field(TH_w(:,:,iblock),tavg_TH,iblock,1)
      endif
      if (accumulate_tavg_now(tavg_TP)) then
        call accumulate_tavg_field(TP_w(:,:,iblock),tavg_TP,iblock,1)
      endif
      if (accumulate_tavg_now(tavg_TZ)) then
        call accumulate_tavg_field(TZ_w(:,:,iblock),tavg_TZ,iblock,1)
      endif
      if (accumulate_tavg_now(tavg_windx)) then
        call accumulate_tavg_field(wind_x1(:,:,iblock),tavg_windx,iblock,1)
      endif
      if (accumulate_tavg_now(tavg_windy)) then
        call accumulate_tavg_field(wind_y1(:,:,iblock),tavg_windy,iblock,1)
      endif
      if (accumulate_tavg_now(tavg_TBV)) then
        do k = 1, k_bv_w
          call accumulate_tavg_field(TBV_w(:,:,k,iblock),  &
                                     tavg_TBV,iblock,k)
        enddo
      endif
    enddo
    
!--------------------------------------------------------------------

 end subroutine mwave_getvar 

!******************************************************************** 
! ROUTINE: mwave_read_restart
! INTERFACE:

    subroutine mwave_read_restart(restart_file, action)

! DESCRIPTION:
! Read auxiliary fields & scalars to restart files

! HISTORY:
! same as module

! Input Parameters:
    character(*), intent(in) :: action
    
! Input/Output Parameters:
    type (datafile), intent (inout) :: restart_file

!--------------------------------------------------------------------
!
!  local variables
!
!--------------------------------------------------------------------
    
    type (io_dim) :: &
      i_dim, j_dim ! dimension descriptors
      
!    type (io_field_desc), save :: mwave_windx, mwave_windy
!    type (io_field_desc), save :: mwave_mask    
!    type (io_field_desc), dimension(num_w*dir_w) :: mwave_spec
    type (io_field_desc), allocatable, save :: mwave_spec(:)
      
    integer (int_kind) :: &
      i, j, k, l,         &! depth loop index
      iblock,             &! block index
      ierr,               &
      n

    character (char_len) :: &
      var_name, long_name    ! for spectrum

    character (len=2) ::    &
      k_char, l_char         ! for spectrum

!--------------------------------------------------------------------
!
!  Add num_w and dir_w to restart file attributes
!
!--------------------------------------------------------------------
    if (trim(action) == 'add_attrib_file') then
     	call add_attrib_file(restart_file, 'num_w', num_w)
     	call add_attrib_file(restart_file, 'dir_w', dir_w) 	
    endif

!--------------------------------------------------------------------
!
!  extract scalars (num_w, dir_w) as file attributes
!  then set runtype to continue
!  and check the num_w and dir_w between namelist and restart file
!--------------------------------------------------------------------
    if (trim(action) == 'extract_attrib_file') then
     	call extract_attrib_file(restart_file, 'num_w', tmp_num_w)
     	call extract_attrib_file(restart_file, 'dir_w', tmp_dir_w) 
     	start_w = 1     ! set runtype to continue run

      if (my_task == master_task) then
    	  if (tmp_num_w /= num_w .or. tmp_dir_w /= dir_w) then
    	  	call exit_POP(sigAbort, 'num_w or dir_w in namelist &
    		                is uninconsistent with restart file')
        endif
      endif
    endif    		          
!--------------------------------------------------------------------
!
!  Define restart file vars
!
!--------------------------------------------------------------------
    if (trim(action) == 'define') then
    	i_dim = construct_io_dim('i', nx_global)
    	j_dim = construct_io_dim('j', ny_global)
        if (.not. allocated(mwave_spec)) then 
           allocate(mwave_spec(num_w*dir_w))
        endif
    	
!    	mwave_mask = construct_io_field('mask_w', i_dim, j_dim,    &
!    	               long_name='mwave mask',       &
!    	               units='m/s', grid_loc='2110',                 &
!    	               field_loc = field_loc_center,                 &
!    	               field_type = field_type_vector,               &
!    	               i2d_array = mask_w)
!    	call data_set (restart_file, 'define', mwave_mask)
!
!    	mwave_windx = construct_io_field('windx_w', i_dim, j_dim,    &
!    	               long_name='10m zonal wind at old time',       &
!    	               units='m/s', grid_loc='2110',                 &
!    	               field_loc = field_loc_center,                 &
!    	               field_type = field_type_vector,               &
!    	               d2d_array = wind_x0)
!    	call data_set (restart_file, 'define', mwave_windx)
!
!    	mwave_windy = construct_io_field('windy_w', i_dim, j_dim,    &
!    	               long_name='10m mer wind at old time',         &
!    	               units='m/s', grid_loc='2110',                 &
!    	               field_loc = field_loc_center,                 &
!    	               field_type = field_type_vector,               &
!    	               d2d_array = wind_y0)
!    	call data_set (restart_file, 'define', mwave_windy)    	
!
!*** assume mwave spectrum as 2-dimesions vars
      n = 0
      do l = 1, dir_w
      	l_char = '00'
      	write(l_char,'(i2.2)') l
      	do k = 1, num_w
      		n = n + 1
      		k_char = '00'
      		write(k_char,'(i2.2)') k
          var_name = char_blank
!      		var_name = 'spec_w_'//trim(k_char)//'_'//trim(l_char)
      		var_name(1:7) = 'spec_w_'
      		var_name(8:9) = trim(k_char)
      		var_name(10:10) = '_'
      		var_name(11:12) = trim(l_char)
!      		long_name = 'mwave spectrum of '//trim(k_char)//' and '  &
!      		            //trim(l_char)//' at current time'
          long_name = char_blank
          long_name(1:18) = 'mwave spectrum of '
          long_name(19:20) = trim(k_char)
          long_name(21:25) = ' and '
          long_name(26:27) = trim(l_char)
          long_name(28:43) = ' at current time'
          mwave_spec(n) = construct_io_field(trim(var_name),       &
                          i_dim, j_dim,                            &
                          long_name = trim(long_name),             &
                          units = 'null',                          &
                          grid_loc='2110',                         &
                          field_loc = field_loc_center,            &
                          field_type = field_type_scalar,          &
                          d2d_array = spec_w(:,:,k,l,:))
      	  call data_set(restart_file, 'define', mwave_spec(n))
      	enddo
      enddo    	
    	
    endif

!--------------------------------------------------------------------
!
!  actually read restart file vars
!
!--------------------------------------------------------------------
    if (trim(action) == 'read') then
!      call data_set (restart_file, 'read', mwave_mask)
!    	call data_set (restart_file, 'read', mwave_windx)
!    	call data_set (restart_file, 'read', mwave_windy)
    	
      n = 0
      do l = 1, dir_w
      	do k = 1, num_w
      		n = n + 1
       	  call data_set(restart_file, 'read', mwave_spec(n))
!          where(mask_w(:,:,:) == 0 ) spec_w(:,:,k,l,:) = 0.0_POP_r8
       	enddo
       enddo   
     endif
    
    if (trim(action) == 'destroy') then
!       call destroy_io_field(mwave_mask)
!    	call destroy_io_field(mwave_windx)
!    	call destroy_io_field(mwave_windy)
    	n = 0
    	do l = 1, dir_w
    		do k = 1, num_w
    			n = n + 1
    	    call destroy_io_field(mwave_spec(n))
    	  enddo
    	enddo
    endif
    
    if (trim(action) == 'update') then
!      call POP_HaloUpdate(mask_w,POP_haloClinic, &
!                          POP_gridHorzLocCenter,            &
!                          POP_fieldKindScalar, errorCode,   &
!                          fillValue = 0_POP_i4)
!      if (errorCode /= POP_Success) then
!      	call POP_ErrorSet(errorCode, &
!      	   'mwave: error updating halo for mask_w')
!      	return
!      endif

!      call POP_HaloUpdate(wind_x0,POP_haloClinic, &
!                          POP_gridHorzLocCenter,            &
!                          POP_fieldKindScalar, errorCode,   &
!                          fillValue = 0.0_POP_r8)
!      if (errorCode /= POP_Success) then
!      	call POP_ErrorSet(errorCode, &
!      	   'mwave: error updating halo for wind_x0')
!      	return
!      endif
!
!      call POP_HaloUpdate(wind_y0,POP_haloClinic, &
!                          POP_gridHorzLocCenter,            &
!                          POP_fieldKindScalar, errorCode,   &
!                          fillValue = 0.0_POP_r8)
!      if (errorCode /= POP_Success) then
!      	call POP_ErrorSet(errorCode, &
!      	   'mwave: error updating halo for wind_y0')
!      	return
!      endif
!    	
      call POP_HaloUpdate(spec_w(:,:,:,:,:),POP_haloClinic, &
                          POP_gridHorzLocCenter,            &
                          POP_fieldKindScalar, errorCode,   &
                          fillValue = 0.0_POP_r8)
      if (errorCode /= POP_Success) then
      	call POP_ErrorSet(errorCode, &
      	   'mwave: error updating halo for spectrum')
      	return
      endif
     endif
!--------------------------------------------------------------------

 end subroutine mwave_read_restart 


!******************************************************************** 
! ROUTINE: mwave_write_restart
! INTERFACE:

    subroutine mwave_write_restart(restart_file, action)

! DESCRIPTION:
! Write auxiliary fields & scalars to restart files

! HISTORY:
! same as module

! Input Parameters:
    character(*), intent(in) :: action
    
! Input/Output Parameters:
    type (datafile), intent (inout) :: restart_file

!--------------------------------------------------------------------
!
!  local variables
!
!--------------------------------------------------------------------
    
    type (io_dim) :: &
      i_dim, j_dim ! dimension descriptors
      
!    type (io_field_desc), save :: mwave_windx, mwave_windy
!    type (io_field_desc), save :: mwave_mask
    
!    type (io_field_desc), dimension(num_w*dir_w) :: mwave_spec
    type (io_field_desc), allocatable, save :: mwave_spec(:)
      
    integer (int_kind) :: &
      i, j, k, l,         &! depth loop index
      iblock,             &! block index
      ierr,               &
      n

    character (char_len) :: &
      var_name, long_name    ! for spectrum

    character (len=2) ::    &
      k_char, l_char         ! for spectrum

!--------------------------------------------------------------------
!
!  Add num_w and dir_w to restart file attributes
!
!--------------------------------------------------------------------
    if (trim(action) == 'add_attrib_file') then
     	call add_attrib_file(restart_file, 'num_w', num_w)
     	call add_attrib_file(restart_file, 'dir_w', dir_w) 	
    endif

!--------------------------------------------------------------------
!
!  Define restart file vars
!
!--------------------------------------------------------------------
    if (trim(action) == 'define') then
    	i_dim = construct_io_dim('i', nx_global)
    	j_dim = construct_io_dim('j', ny_global)
        if (.not. allocated(mwave_spec)) then 
           allocate(mwave_spec(num_w*dir_w))
        endif
    	
!    	mwave_mask = construct_io_field('mask_w', i_dim, j_dim,    &
!    	               long_name='mwave mask',       &
!    	               units='m/s', grid_loc='2110',                 &
!    	               field_loc = field_loc_center,                 &
!    	               field_type = field_type_vector,               &
!    	               i2d_array = mask_w)
!    	call data_set (restart_file, 'define', mwave_mask)
!
!    	mwave_windx = construct_io_field('windx_w', i_dim, j_dim,    &
!    	               long_name='10m zonal wind at old time',       &
!    	               units='m/s', grid_loc='2110',                 &
!    	               field_loc = field_loc_center,                 &
!    	               field_type = field_type_vector,               &
!    	               d2d_array = wind_x0)
!    	call data_set (restart_file, 'define', mwave_windx)
!
!    	mwave_windy = construct_io_field('windy_w', i_dim, j_dim,    &
!    	               long_name='10m mer wind at old time',         &
!    	               units='m/s', grid_loc='2110',                 &
!    	               field_loc = field_loc_center,                 &
!    	               field_type = field_type_vector,               &
!    	               d2d_array = wind_y0)
!    	call data_set (restart_file, 'define', mwave_windy)    	
!
!*** assume mwave spectrum as 2-dimesions vars
      n = 0
      do l = 1, dir_w
      	l_char = '00'
      	write(l_char,'(i2.2)') l
!      	print *, 'l_char', l_char
      	do k = 1, num_w
      		n = n + 1
!      		print *, 'num_w*dir_w ', n
      		k_char = '00'
      		write(k_char,'(i2.2)') k
!      		print *, 'k_char', k_char
      		var_name = char_blank
!          var_name = 'spec_w_'//k_char//'_'//l_char
      		var_name(1:7) = 'spec_w_'
      		var_name(8:9) = trim(k_char)
      		var_name(10:10) = '_'
      		var_name(11:12) = trim(l_char)
!      		print *, 'var_name', trim(var_name)
      		long_name = char_blank
!      		long_name = 'mwave spectrum of '//trim(k_char)//' and '  &
!      		            //trim(l_char)//' at current time'
          long_name(1:18) = 'mwave spectrum of '
          long_name(19:20) = trim(k_char)
          long_name(21:25) = ' and '
          long_name(26:27) = trim(l_char)
          long_name(28:43) = ' at current time'
          mwave_spec(n) = construct_io_field(trim(var_name),       &
                          i_dim, j_dim,                            &
                          long_name = trim(long_name),             &
                          units = 'null',                          &
                          grid_loc='2110',                         &
                          field_loc = field_loc_center,            &
                          field_type = field_type_scalar,          &
                          d2d_array = spec_w(:,:,k,l,:))
      	  call data_set(restart_file, 'define', mwave_spec(n))
      	enddo
      enddo    	
    	
    endif

!--------------------------------------------------------------------
!
!  Write restart file vars
!
!--------------------------------------------------------------------
    if (trim(action) == 'write') then
!      call data_set (restart_file, 'write', mwave_mask)
!    	call data_set (restart_file, 'write', mwave_windx)
!    	call data_set (restart_file, 'write', mwave_windy)
    	
      n = 0
      do l = 1, dir_w
      	do k = 1, num_w
      		n = n + 1
       	  call data_set(restart_file, 'write', mwave_spec(n))
       	enddo
       enddo   
     endif
    
!--------------------------------------------------------------------

 end subroutine mwave_write_restart 
 
end
